package reservations

import (
	"database/sql"
	"fmt"
	"strings"

	cdb "code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/errx"
	. "code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	sqlSelectReservations = `SELECT` +
		` login,` +
		` type,` +
		` reason,` +
		` expires_on` +
		` FROM login_reservations` +
		` WHERE`
)

func (c *reservationDBImpl) GetReservation(ctx context.Context, login string) (*models.ReservationProperties, error) {
	querySuffix := fmt.Sprintf("login IN ($1)")
	query := strings.Join([]string{sqlSelectReservations, querySuffix}, " ")

	row := c.mdb.QueryRow(ctx, "login_reservation", query, login)

	return GetReservationPropertiesFromRow(row)
}

func (c *reservationDBImpl) GetReservations(ctx context.Context, logins []string) ([]models.ReservationProperties, error) {
	var querySuffix string
	var queryParams []interface{}
	var idSuffix []string

	for i, id := range logins {
		idSuffix = append(idSuffix, fmt.Sprintf("$%d", i+1))
		queryParams = append(queryParams, id)
	}
	querySuffix = fmt.Sprintf("login IN (%s)", strings.Join(idSuffix, ","))

	query := strings.Join([]string{sqlSelectReservations, querySuffix}, " ")

	rows, err := c.mdb.Query(ctx, "login_reservations", query, queryParams...)
	if err != nil {
		return nil, errx.New(err, errx.Fields{"num_logins": len(logins)})
	}
	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbResults, err := GetReservationPropertiesFromRows(rows)
	if err != nil {
		return nil, err
	}

	return dbResults, nil
}

func GetReservationPropertiesFromRow(row cdb.Row) (*models.ReservationProperties, error) {
	up := models.ReservationProperties{}

	err := row.Scan(
		&up.Login,
		&up.Type,
		&up.Reason,
		&up.ExpiresOn,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, errx.New(ErrNoProperties)
	case err != nil:
		return nil, errx.New(err)
	default:
	}

	return &up, nil
}

func GetReservationPropertiesFromRows(rows database.Rows) ([]models.ReservationProperties, error) {
	results := []models.ReservationProperties{}

	for rows.Next() {
		up, err := GetReservationPropertiesFromRow(rows)
		if err != nil {
			return nil, err
		}
		results = append(results, *up)
	}

	err := rows.Err()
	if err != nil {
		return nil, errx.New(err)
	}

	return results, nil
}
