package reservations

import (
	"fmt"

	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"github.com/afex/hystrix-go/hystrix"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"Reservations.AddReservation": {
			Timeout:               1000,
			MaxConcurrentRequests: 200,
		},
		"Reservations.DeleteReservation": {
			Timeout:               1000,
			MaxConcurrentRequests: 200,
		},
		"Reservations.GetReservation": {
			Timeout:               1000,
			MaxConcurrentRequests: 200,
		},
		"Reservations.GetReservations": {
			Timeout:               1000,
			MaxConcurrentRequests: 200,
		},
		"Reservations.UpdateReservation": {
			Timeout:               1000,
			MaxConcurrentRequests: 200,
		},
	})
}

// HystrixReservationsBackend wraps hystrix circuits around each method.
type HystrixReservationsBackend struct {
	Backend Backend
}
type reservationsBackendTripSkipper interface {
	SkipTrip() bool
}

// GetReservations calls Backend.GetReservations within a hystrix circuit named Reservations.GetReservations.
func (b *HystrixReservationsBackend) GetReservations(ctx context.Context, logins []string) (out0 []models.ReservationProperties, err error) {
	type response struct {
		out0 []models.ReservationProperties
		err  error
	}

	ch := make(chan response, 1)

	cCtx, cancel := context.WithCancel(ctx)

	hystrixErr := hystrix.Do("Reservations.GetReservations", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Reservations.GetReservations circuit panic=%v", r))
			}
		}()

		out0, err := b.Backend.GetReservations(cCtx, logins)

		ch <- response{
			out0: out0,
			err:  err,
		}

		skipTripper, ok := errx.Unwrap(err).(reservationsBackendTripSkipper)
		if ok && skipTripper.SkipTrip() {
			return nil
		}

		return err
	}, nil)

	cancel()

	if hystrixErr != nil {
		return
	}

	resp := <-ch
	out0 = resp.out0
	err = resp.err

	return
}

// GetReservation calls Backend.GetReservation within a hystrix circuit named Reservations.GetReservation.
func (b *HystrixReservationsBackend) GetReservation(ctx context.Context, login string) (out0 *models.ReservationProperties, err error) {
	type response struct {
		out0 *models.ReservationProperties
		err  error
	}

	ch := make(chan response, 1)

	cCtx, cancel := context.WithCancel(ctx)

	hystrixErr := hystrix.Do("Reservations.GetReservation", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Reservations.GetReservation circuit panic=%v", r))
			}
		}()

		out0, err := b.Backend.GetReservation(cCtx, login)

		ch <- response{
			out0: out0,
			err:  err,
		}

		skipTripper, ok := errx.Unwrap(err).(reservationsBackendTripSkipper)
		if ok && skipTripper.SkipTrip() {
			return nil
		}

		return err
	}, nil)

	cancel()

	if hystrixErr != nil {
		return
	}

	resp := <-ch
	out0 = resp.out0
	err = resp.err

	return
}

// AddReservation calls Backend.AddReservation within a hystrix circuit named Reservations.AddReservation.
func (b *HystrixReservationsBackend) AddReservation(ctx context.Context, prop models.ReservationProperties) (err error) {
	type response struct {
		err error
	}

	ch := make(chan response, 1)

	cCtx, cancel := context.WithCancel(ctx)

	hystrixErr := hystrix.Do("Reservations.AddReservation", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Reservations.AddReservation circuit panic=%v", r))
			}
		}()

		err := b.Backend.AddReservation(cCtx, prop)

		ch <- response{
			err: err,
		}

		skipTripper, ok := errx.Unwrap(err).(reservationsBackendTripSkipper)
		if ok && skipTripper.SkipTrip() {
			return nil
		}

		return err
	}, nil)

	cancel()

	if hystrixErr != nil {
		return
	}

	resp := <-ch
	err = resp.err

	return
}

// DeleteReservation calls Backend.DeleteReservation within a hystrix circuit named Reservations.DeleteReservation.
func (b *HystrixReservationsBackend) DeleteReservation(ctx context.Context, login string) (err error) {
	type response struct {
		err error
	}

	ch := make(chan response, 1)

	cCtx, cancel := context.WithCancel(ctx)

	hystrixErr := hystrix.Do("Reservations.DeleteReservation", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Reservations.DeleteReservation circuit panic=%v", r))
			}
		}()

		err := b.Backend.DeleteReservation(cCtx, login)

		ch <- response{
			err: err,
		}

		skipTripper, ok := errx.Unwrap(err).(reservationsBackendTripSkipper)
		if ok && skipTripper.SkipTrip() {
			return nil
		}

		return err
	}, nil)

	cancel()

	if hystrixErr != nil {
		return
	}

	resp := <-ch
	err = resp.err

	return
}

// UpdateReservation calls Backend.UpdateReservation within a hystrix circuit named Reservations.UpdateReservation.
func (b *HystrixReservationsBackend) UpdateReservation(ctx context.Context, prop models.ReservationProperties) (err error) {
	type response struct {
		err error
	}

	ch := make(chan response, 1)

	cCtx, cancel := context.WithCancel(ctx)

	hystrixErr := hystrix.Do("Reservations.UpdateReservation", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Reservations.UpdateReservation circuit panic=%v", r))
			}
		}()

		err := b.Backend.UpdateReservation(cCtx, prop)

		ch <- response{
			err: err,
		}

		skipTripper, ok := errx.Unwrap(err).(reservationsBackendTripSkipper)
		if ok && skipTripper.SkipTrip() {
			return nil
		}

		return err
	}, nil)

	cancel()

	if hystrixErr != nil {
		return
	}

	resp := <-ch
	err = resp.err

	return
}
