package reservations

import (
	"context"
	"errors"
	"testing"

	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/database/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestUpdateReservation(t *testing.T) {
	login := "reserved_login"
	reservationType := "JTV Reservation"
	reason := "test"
	err := errors.New("db connection error")
	expire := time.Now()
	prop := models.ReservationProperties{
		Login:     login,
		Type:      reservationType,
		Reason:    reason,
		ExpiresOn: &expire,
	}
	for _, scenario := range []struct {
		Name  string
		Setup func(t *testing.T, mdb *mocks.Querier)
		Prop  models.ReservationProperties
		Error error
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, mdb *mocks.Querier) {
				mdb.On("Exec", mock.Anything, "update_reservation", sqlUpdateReservations, []interface{}{
					login,
					reservationType,
					reason,
					expire,
				}).Return(&mocks.Result{}, nil)
			},
			Prop:  prop,
			Error: nil,
		},
		{
			Name: "happy path with nil expire time",
			Setup: func(t *testing.T, mdb *mocks.Querier) {
				mdb.On("Exec", mock.Anything, "update_reservation", sqlUpdateReservations, []interface{}{
					login,
					reservationType,
					reason,
					nil,
				}).Return(&mocks.Result{}, nil)
			},
			Prop: models.ReservationProperties{
				Login:     login,
				Type:      reservationType,
				Reason:    reason,
				ExpiresOn: nil,
			},
			Error: nil,
		},
		{
			Name: "db returns error",
			Setup: func(t *testing.T, mdb *mocks.Querier) {
				mdb.On("Exec", mock.Anything, "update_reservation", sqlUpdateReservations, []interface{}{
					login,
					reservationType,
					reason,
					expire,
				}).Return(&mocks.Result{}, err)
			},
			Prop:  prop,
			Error: err,
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			mdb := &mocks.Querier{}
			reservationDB := reservationDBImpl{mdb: mdb}

			if scenario.Setup != nil {
				scenario.Setup(t, mdb)
			}

			err := reservationDB.UpdateReservation(context.Background(), scenario.Prop)
			assert.Equal(t, errx.Unwrap(err), scenario.Error)
		})
	}
}

func TestAddReservation(t *testing.T) {
	login := "reserved_login"
	reservationType := "JTV Reservation"
	reason := "test"
	err := errors.New("db connection error")
	expire := time.Now()
	prop := models.ReservationProperties{
		Login:     login,
		Type:      reservationType,
		Reason:    reason,
		ExpiresOn: &expire,
	}
	for _, scenario := range []struct {
		Name  string
		Setup func(t *testing.T, mdb *mocks.Querier)
		Prop  models.ReservationProperties
		Error error
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, mdb *mocks.Querier) {
				mdb.On("Exec", mock.Anything, "add_reservation", sqlInsertReservations, []interface{}{
					login,
					reservationType,
					reason,
					expire,
				}).Return(&mocks.Result{}, nil)
			},
			Prop:  prop,
			Error: nil,
		},
		{
			Name: "happy path with nil expire time",
			Setup: func(t *testing.T, mdb *mocks.Querier) {
				mdb.On("Exec", mock.Anything, "add_reservation", sqlInsertReservations, []interface{}{
					login,
					reservationType,
					reason,
					nil,
				}).Return(&mocks.Result{}, nil)
			},
			Prop: models.ReservationProperties{
				Login:     login,
				Type:      reservationType,
				Reason:    reason,
				ExpiresOn: nil,
			},
			Error: nil,
		},
		{
			Name: "db returns error",
			Setup: func(t *testing.T, mdb *mocks.Querier) {
				mdb.On("Exec", mock.Anything, "add_reservation", sqlInsertReservations, []interface{}{
					login,
					reservationType,
					reason,
					expire,
				}).Return(&mocks.Result{}, err)
			},
			Prop:  prop,
			Error: err,
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			mdb := &mocks.Querier{}
			reservationDB := reservationDBImpl{mdb: mdb}

			if scenario.Setup != nil {
				scenario.Setup(t, mdb)
			}

			err := reservationDB.AddReservation(context.Background(), scenario.Prop)
			assert.Equal(t, errx.Unwrap(err), scenario.Error)
		})
	}
}

func TestDeleteReservation(t *testing.T) {
	login := "reserved_login"
	err := errors.New("db connection error")
	for _, scenario := range []struct {
		Name  string
		Setup func(t *testing.T, mdb *mocks.Querier)
		Error error
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, mdb *mocks.Querier) {
				mdb.On("Exec", mock.Anything, "delete_reservation", sqlDeleteReservation, []interface{}{
					login,
				}).Return(&mocks.Result{}, nil)
			},
			Error: nil,
		},
		{
			Name: "db returns error",
			Setup: func(t *testing.T, mdb *mocks.Querier) {
				mdb.On("Exec", mock.Anything, "delete_reservation", sqlDeleteReservation, []interface{}{
					login,
				}).Return(&mocks.Result{}, err)
			},
			Error: err,
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			mdb := &mocks.Querier{}
			reservationDB := reservationDBImpl{mdb: mdb}

			if scenario.Setup != nil {
				scenario.Setup(t, mdb)
			}

			err := reservationDB.DeleteReservation(context.Background(), login)
			assert.Equal(t, errx.Unwrap(err), scenario.Error)
		})
	}
}
