package backend

import (
	"fmt"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database"
)

type Tx struct {
	db.Tx
	name string
}

func OpenTx(ctx context.Context, q database.Querier, name string) (*Tx, error) {
	tx, err := q.Begin(ctx, fmt.Sprintf("%s_transaction", strings.Join(strings.Split(name, " "), "_")))
	if err != nil {
		return nil, errx.New(err, errx.Fields{util.DBError: "Error opening transaction"})
	}

	return &Tx{
		Tx:   tx,
		name: name,
	}, nil
}

func (tx *Tx) Close(ctx context.Context, err error) {
	if err != nil {
		if txErr := tx.Rollback(); txErr != nil {
			msg := fmt.Sprintf("Error rolling back %s transaction", tx.name)
			txErrMsg := fmt.Sprintf("%v", txErr)
			errMsg := fmt.Sprintf("%v", err)
			logx.Error(ctx, fmt.Sprintf("%s (%q): %s", msg, errMsg, txErrMsg), logx.Fields{util.DBError: fmt.Sprintf("%s: %v", msg, txErr), "tx_error": txErrMsg, "error": errMsg})
			return
		}
		return
	}

	if err = tx.Commit(); err != nil {
		msg := fmt.Sprintf("Error committing %s transaction", tx.name)
		errMsg := fmt.Sprintf("%v", err)
		logx.Error(ctx, fmt.Sprintf("%s: %s", msg, errMsg), logx.Fields{util.DBError: msg, "error": errMsg})
	}
}
