package cacheddb

import (
	"fmt"
	"runtime"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/users"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/models"
)

type backendImpl struct {
	db    users.Backend
	cache users.CacheBackend
}

func NewBackend(db users.Backend, cache users.CacheBackend) (users.Backend, error) {
	return &backendImpl{
		db:    db,
		cache: cache,
	}, nil
}

func getCaller() string {
	_, filename, line, ok := runtime.Caller(1)
	if ok {
		return fmt.Sprintf("%s:%d", filename, line)
	}

	return "undefined"
}

func (c *backendImpl) UpdateProperties(ctx context.Context, uup *models.UpdateableProperties, cup *models.Properties) error {
	err := c.db.UpdateProperties(ctx, uup, cup)
	if err != nil {
		return err
	}

	go func() {
		bgCtx := backend.DetachContext(ctx)
		err := c.cache.ExpireUserProperties(bgCtx, cup)
		if err != nil {
			logx.Error(bgCtx, err)
		}
	}()

	return nil
}

func (c *backendImpl) GetUserPropertiesByDisplayname(ctx context.Context, dn *string) (prop *models.Properties, err error) {
	defer func() {
		backend.ReportIdentifiers(ctx, "SingleDisplayname", 1, err)
	}()

	prop, cacheErr := c.cache.GetUserProperties(ctx, models.UserDisplaynameCacheKey, *dn)
	if cacheErr != nil {
		if !utils.IsKnownErr(cacheErr) {
			logx.Error(ctx, cacheErr)
		}
	} else if prop != nil {
		return
	}

	prop, err = c.db.GetUserPropertiesByDisplayname(ctx, dn)
	if err != nil {
		return
	}

	if prop != nil {
		c.bgCacheProperties(ctx, cacheErr, false, *prop)
	}

	return
}

func (c *backendImpl) bgCacheProperties(ctx context.Context, cacheErr error, overwrite bool, props ...models.Properties) {
	if utils.IsHystrixErr(cacheErr) {
		return
	}

	bgCtx := backend.DetachContext(ctx)
	caller := getCaller()
	go func() {
		err := c.cache.CacheUsersProperties(bgCtx, props, overwrite)
		if err != nil && !utils.IsKnownErr(err) {
			logx.Error(bgCtx, "Background caching user properties error", logx.Fields{
				"error":     err,
				"props":     len(props),
				"overwrite": overwrite,
				"caller":    caller,
			})
		}
	}()
}

func (c *backendImpl) overwriteCache(ctx context.Context, id string) error {
	props, err := c.db.GetUserPropertiesByID(ctx, id, util.ReadOptions{ReadFromMaster: true})
	if err != nil {
		return err
	}

	if props == nil {
		return errx.New(fmt.Sprintf("User %s missing in master db during cache overwrite.", id))
	}

	err = c.cache.CacheUsersProperties(ctx, []models.Properties{*props}, true)
	return err
}

func (c *backendImpl) GetUserPropertiesByID(ctx context.Context, idString string, opts util.ReadOptions) (up *models.Properties, err error) {
	defer func() {
		backend.ReportIdentifiers(ctx, "SingleID", 1, err)
	}()

	var cacheErr error
	if !opts.ReadFromMaster {
		up, cacheErr = c.cache.GetUserProperties(ctx, models.UserIDCacheKey, idString)
		if cacheErr != nil {
			if !utils.IsKnownErr(cacheErr) {
				logx.Error(ctx, cacheErr)
			}
		} else if up != nil {
			return
		}
	}

	up, err = c.db.GetUserPropertiesByID(ctx, idString, opts)
	if err != nil {
		return
	}

	if up != nil {
		c.bgCacheProperties(ctx, cacheErr, opts.OverwriteCache, *up)
	}

	return
}

func (c *backendImpl) GetUserPropertiesByLogin(ctx context.Context, login string, opts util.ReadOptions) (up *models.Properties, err error) {
	defer func() {
		backend.ReportIdentifiers(ctx, "SingleLogin", 1, err)
	}()

	var cacheErr error
	if !opts.ReadFromMaster {
		up, cacheErr = c.cache.GetUserProperties(ctx, models.UserLoginCacheKey, login)
		if cacheErr != nil {
			if !utils.IsKnownErr(cacheErr) {
				logx.Error(ctx, cacheErr)
			}
		} else if up != nil {
			return
		}
	}

	up, err = c.db.GetUserPropertiesByLogin(ctx, login, opts)
	if err != nil {
		return
	}

	if up != nil {
		c.bgCacheProperties(ctx, cacheErr, opts.OverwriteCache, *up)
	}

	return
}

func (c *backendImpl) GetUsersProperties(ctx context.Context, field string, identifiers []string) ([]models.Properties, error) {
	cachedUps, missingKeyIndexes, cacheErr := c.cache.GetUsersProperties(ctx, field, identifiers)
	if cacheErr != nil {
		if !utils.IsKnownErr(cacheErr) {
			logx.Error(ctx, cacheErr)
		}
		missingKeyIndexes = []int{}
		cachedUps = []models.Properties{}
		for i := range identifiers {
			missingKeyIndexes = append(missingKeyIndexes, i)
		}
	}

	if len(missingKeyIndexes) == 0 {
		return cachedUps, nil
	}

	missingKeys := []string{}
	for _, mIdx := range missingKeyIndexes {
		missingKeys = append(missingKeys, identifiers[mIdx])
	}

	dbResults, err := c.db.GetUsersProperties(ctx, field, missingKeys)
	if err != nil {
		return nil, err
	}

	if len(dbResults) == 0 {
		return cachedUps, nil
	}

	c.bgCacheProperties(ctx, cacheErr, false, dbResults...)

	total := append(cachedUps, dbResults...)
	return total, nil

}

func (c *backendImpl) GetUserPropertiesBulk(ctx context.Context, params *models.FilterParams) (ups []models.Properties, err error) {
	start := time.Now()
	defer func() {
		dur := time.Since(start)
		backend.ReportBulkIdentifiers(ctx, "BulkID", len(params.IDs), dur, err)
		backend.ReportBulkIdentifiers(ctx, "BulkLogin", len(params.Logins), dur, err)
		backend.ReportBulkIdentifiers(ctx, "BulkEmail", len(params.Emails), dur, err)
		backend.ReportBulkIdentifiers(ctx, "BulkDisplayname", len(params.DisplayNames), dur, err)
		backend.ReportBulkIdentifiers(ctx, "BulkRemoteIP", len(params.Ips), dur, err)
	}()

	upsc := make(chan []models.Properties)
	errc := make(chan error)

	fis := []util.Fieldidentifier{
		{Field: "u.id", Ids: params.IDs},
		{Field: "login", Ids: params.Logins},
		{Field: "lower(email)", Ids: params.Emails},
		{Field: "lower(displayname)", Ids: params.DisplayNames},
		{Field: "remote_ip", Ids: params.Ips},
	}

	wait := 0
	for _, fi := range fis {
		if len(fi.Ids) > 0 {
			wait++
			go func(fi util.Fieldidentifier) {
				ups, err := c.GetUsersProperties(ctx, fi.Field, fi.Ids)
				if err != nil {
					errc <- err
					return
				}
				upsc <- ups
			}(fi)
		}
	}

	for i := 0; i < wait; i++ {
		select {
		case upsres := <-upsc:
			if upsres != nil {
				ups = append(ups, upsres...)
			}
		case err = <-errc:
			return
		}
	}
	return
}

func (c *backendImpl) GetLogins(ctx context.Context, logins []string) ([]models.LoginProperties, error) {
	return c.db.GetLogins(ctx, logins)
}

func (c *backendImpl) GetUserPropertiesLike(ctx context.Context, field string, pattern string) ([]models.Properties, error) {
	return c.db.GetUserPropertiesLike(ctx, field, pattern)
}

func (c *backendImpl) AlterDMCAStrike(ctx context.Context, id string, delta int) error {
	err := c.db.AlterDMCAStrike(ctx, id, delta)
	if err != nil {
		return err
	}
	cup, err := c.db.GetUserPropertiesByID(ctx, id, util.ReadOptions{ReadFromMaster: false})
	if err != nil {
		return err
	}
	return c.cache.ExpireUserProperties(ctx, cup)
}

func (c *backendImpl) BanUser(ctx context.Context, id string, warn bool, tosBan bool, dmcaStrikes int, tosStrikes int) error {
	err := c.db.BanUser(ctx, id, warn, tosBan, dmcaStrikes, tosStrikes)
	if err != nil {
		return err
	}

	cup, err := c.db.GetUserPropertiesByID(ctx, id, util.ReadOptions{ReadFromMaster: false})
	if err != nil {
		return err
	}
	return c.cache.ExpireUserProperties(ctx, cup)
}

func (c *backendImpl) UnbanUser(ctx context.Context, id string, tosCount, dmcaCount int) error {
	err := c.db.UnbanUser(ctx, id, tosCount, dmcaCount)
	if err != nil {
		return err
	}

	cup, err := c.db.GetUserPropertiesByID(ctx, id, util.ReadOptions{ReadFromMaster: false})
	if err != nil {
		return err
	}

	return c.cache.ExpireUserProperties(ctx, cup)
}

func (c *backendImpl) GetBannedUsers(ctx context.Context, until time.Time) ([]models.Properties, error) {
	return c.db.GetBannedUsers(ctx, until)
}

func (c *backendImpl) GetUserBlock(ctx context.Context, login string) (*models.BlockProperties, error) {
	return c.db.GetUserBlock(ctx, login)
}

func (c *backendImpl) GetUserImages(ctx context.Context, id string, login string) (*models.ImageProperties, error) {
	return c.db.GetUserImages(ctx, id, login)
}

func (c *backendImpl) CreateUser(ctx context.Context, up *models.CreateUserProperties) error {
	return c.db.CreateUser(ctx, up)
}

func (c *backendImpl) CreateUserEmail(ctx context.Context, id, code string) error {
	return c.db.CreateUserEmail(ctx, id, code)
}

func (c *backendImpl) GetUserPhoneNumber(ctx context.Context, id string) (*models.PhoneNumberProperties, error) {
	return c.db.GetUserPhoneNumber(ctx, id)
}

func (c *backendImpl) VerifyUserPhoneNumber(ctx context.Context, id string) error {
	if err := c.db.VerifyUserPhoneNumber(ctx, id); err != nil {
		return err
	}

	prop, err := c.db.GetUserPropertiesByID(ctx, id, util.ReadOptions{ReadFromMaster: false})
	if err != nil {
		return err
	}
	return c.cache.ExpireUserProperties(ctx, prop)
}

func (c *backendImpl) GetGlobalPrivilegedUsers(ctx context.Context, roles []string) (*models.GlobalPrivilegedUsers, error) {
	return c.db.GetGlobalPrivilegedUsers(ctx, roles)
}

func (c *backendImpl) SetUserImageProperties(ctx context.Context, updates models.ImageProperties) error {
	err := c.db.SetUserImageProperties(ctx, updates)
	if err != nil {
		return err
	}

	cup, err := c.db.GetUserPropertiesByID(ctx, updates.ID, util.ReadOptions{ReadFromMaster: false})
	if err != nil {
		return err
	}
	return c.cache.ExpireUserProperties(ctx, cup)
}

func (c *backendImpl) HardDeleteUser(ctx context.Context, ID string, skipBlock bool) (*models.Properties, error) {
	// Delete user
	props, err := c.db.HardDeleteUser(ctx, ID, skipBlock)
	if err != nil {
		return nil, err
	}

	// Overwrite from cache
	return props, c.overwriteCache(ctx, ID)
}

func (c *backendImpl) SoftDeleteUser(ctx context.Context, ID string) error {
	// Delete user
	err := c.db.SoftDeleteUser(ctx, ID)
	if err != nil {
		return err
	}

	// Overwrite from cache
	return c.overwriteCache(ctx, ID)
}

func (c *backendImpl) UndeleteUser(ctx context.Context, ID string) error {
	// Delete user
	err := c.db.UndeleteUser(ctx, ID)
	if err != nil {
		return err
	}

	// Overwrite from cache
	return c.overwriteCache(ctx, ID)
}
