package cacheddb

import (
	"testing"

	"time"

	"code.justin.tv/web/users-service/backend/users"
	"code.justin.tv/web/users-service/backend/users/mocks"
	"code.justin.tv/web/users-service/backend/util"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type BackendTest struct {
	suite.Suite
	mockCache   *mocks.CacheBackend
	mockBackend *mocks.Backend
	backend     users.Backend
	users       map[string]*models.Properties
	readOpt     util.ReadOptions
}

func (suite *BackendTest) SetupTest() {
	var err error
	suite.mockCache = &mocks.CacheBackend{}
	suite.mockBackend = &mocks.Backend{}
	suite.backend, err = NewBackend(suite.mockBackend, suite.mockCache)
	assert.NoError(suite.T(), err)
	suite.readOpt = util.ReadOptions{ReadFromMaster: false, OverwriteCache: false}
}

func (suite *BackendTest) TestGetUserPropertiesByDisplaynameWithCache() {
	suite.mockCache.On("GetUserProperties", mock.Anything, "displayname", *Users[AdminUserID].Displayname).Return(Users[AdminUserID], nil)
	suite.mockBackend.On("GetUserPropertiesByDisplayname", mock.Anything, Users[AdminUserID].Displayname).Return(Users[AdminUserID], nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, []models.Properties{*Users[AdminUserID]}, false).Return(nil)

	up, err := suite.backend.GetUserPropertiesByDisplayname(context.Background(), Users[AdminUserID].Displayname)

	assert.NoError(suite.T(), err)
	suite.mockCache.AssertCalled(suite.T(), "GetUserProperties", mock.Anything, mock.Anything, mock.Anything)
	suite.mockBackend.AssertNotCalled(suite.T(), "GetUserPropertiesByDisplayname", mock.Anything, mock.Anything)
	suite.mockCache.AssertNotCalled(suite.T(), "CacheUsersProperties", mock.Anything, mock.Anything, mock.Anything)
	assert.Equal(suite.T(), up, Users[AdminUserID])
}

func (suite *BackendTest) TestGetUserPropertiesByDisplaynameWithoutCache() {
	suite.mockCache.On("GetUserProperties", mock.Anything, "displayname", *Users[AdminUserID].Displayname).Return(nil, nil)
	suite.mockBackend.On("GetUserPropertiesByDisplayname", mock.Anything, Users[AdminUserID].Displayname).Return(Users[AdminUserID], nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, []models.Properties{*Users[AdminUserID]}, false).Return(nil)

	up, err := suite.backend.GetUserPropertiesByDisplayname(context.Background(), Users[AdminUserID].Displayname)

	assert.NoError(suite.T(), err)
	suite.mockBackend.AssertExpectations(suite.T())
	assert.Equal(suite.T(), up, Users[AdminUserID])
}

func (suite *BackendTest) TestGetUserPropertiesByIDWithCache() {
	suite.mockCache.On("GetUserProperties", mock.Anything, "u.id", AdminUserID).Return(Users[AdminUserID], nil)
	suite.mockBackend.On("GetUserPropertiesByID", mock.Anything, AdminUserID, suite.readOpt).Return(Users[AdminUserID], nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, []models.Properties{*Users[AdminUserID]}, false).Return(nil)

	up, err := suite.backend.GetUserPropertiesByID(context.Background(), AdminUserID, suite.readOpt)

	assert.NoError(suite.T(), err)
	suite.mockCache.AssertCalled(suite.T(), "GetUserProperties", mock.Anything, mock.Anything, mock.Anything)
	suite.mockBackend.AssertNotCalled(suite.T(), "GetUserPropertiesByID", mock.Anything, mock.Anything, mock.Anything)
	suite.mockCache.AssertNotCalled(suite.T(), "CacheUsersProperties", mock.Anything, mock.Anything, mock.Anything)
	assert.Equal(suite.T(), up, Users[AdminUserID])
}

func (suite *BackendTest) TestGetUserPropertiesByIDWithoutCache() {
	suite.mockCache.On("GetUserProperties", mock.Anything, "u.id", AdminUserID).Return(nil, nil)
	suite.mockBackend.On("GetUserPropertiesByID", mock.Anything, AdminUserID, suite.readOpt).Return(Users[AdminUserID], nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, []models.Properties{*Users[AdminUserID]}, false).Return(nil)

	up, err := suite.backend.GetUserPropertiesByID(context.Background(), AdminUserID, suite.readOpt)

	assert.NoError(suite.T(), err)
	suite.mockBackend.AssertExpectations(suite.T())
	assert.Equal(suite.T(), up, Users[AdminUserID])
}

func (suite *BackendTest) TestGetUsersPropertiesWithAllCache() {
	identifiers := []string{AdminUserID, ToBanUserID}
	props := []models.Properties{*Users[AdminUserID], *Users[ToBanUserID]}
	suite.mockCache.On("GetUsersProperties", mock.Anything, "u.id", identifiers).Return(props, []int{}, nil)
	suite.mockBackend.On("GetUsersProperties", mock.Anything, "u.id", identifiers).Return(props, nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, mock.Anything, mock.Anything).Return(nil)

	ups, err := suite.backend.GetUsersProperties(context.Background(), "u.id", identifiers)

	assert.NoError(suite.T(), err)
	suite.mockCache.AssertCalled(suite.T(), "GetUsersProperties", mock.Anything, mock.Anything, mock.Anything)
	suite.mockBackend.AssertNotCalled(suite.T(), "GetUsersProperties", mock.Anything, mock.Anything, mock.Anything)
	suite.mockCache.AssertNotCalled(suite.T(), "CacheUsersProperties", mock.Anything, mock.Anything, mock.Anything)
	assert.Equal(suite.T(), ups, props)
}

func (suite *BackendTest) TestGetUsersPropertiesWithoutCache() {
	identifiers := []string{AdminUserID, ToBanUserID}
	props := []models.Properties{*Users[AdminUserID], *Users[ToBanUserID]}
	suite.mockCache.On("GetUsersProperties", mock.Anything, "u.id", identifiers).Return([]models.Properties{}, []int{0, 1}, nil)
	suite.mockBackend.On("GetUsersProperties", mock.Anything, "u.id", identifiers).Return(props, nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, mock.Anything, mock.Anything).Return(nil)

	ups, err := suite.backend.GetUsersProperties(context.Background(), "u.id", identifiers)

	assert.NoError(suite.T(), err)
	suite.mockBackend.AssertExpectations(suite.T())
	assert.Equal(suite.T(), ups, props)
}

func (suite *BackendTest) TestGetUsersPropertiesWithMixedSource() {
	identifiers := []string{AdminUserID, ToBanUserID}
	props := []models.Properties{*Users[AdminUserID], *Users[ToBanUserID]}
	suite.mockCache.On("GetUsersProperties", mock.Anything, "u.id", identifiers).Return([]models.Properties{*Users[AdminUserID]}, []int{1}, nil)
	suite.mockBackend.On("GetUsersProperties", mock.Anything, "u.id", []string{ToBanUserID}).Return([]models.Properties{*Users[ToBanUserID]}, nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, mock.Anything, mock.Anything).Return(nil)

	ups, err := suite.backend.GetUsersProperties(context.Background(), "u.id", identifiers)

	assert.NoError(suite.T(), err)
	suite.mockBackend.AssertExpectations(suite.T())
	assert.Equal(suite.T(), ups, props)
}

func (suite *BackendTest) TestGetUserPropertiesByIDWithOverwroteCache() {
	suite.mockCache.On("GetUserProperties", mock.Anything, "u.id", AdminUserID).Return(Users[AdminUserID], nil)
	suite.mockBackend.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return(Users[AdminUserID], nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, []models.Properties{*Users[AdminUserID]}, true).Return(nil)

	up, err := suite.backend.GetUserPropertiesByID(context.Background(), AdminUserID, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true})

	time.Sleep(2 * time.Second)
	assert.NoError(suite.T(), err)
	suite.mockBackend.AssertCalled(suite.T(), "GetUserPropertiesByID", mock.Anything, mock.Anything, mock.Anything)
	suite.mockCache.AssertCalled(suite.T(), "CacheUsersProperties", mock.Anything, mock.Anything, mock.Anything)
	assert.Equal(suite.T(), up, Users[AdminUserID])
}

func (suite *BackendTest) TestUpdateProperties() {
	email := "new_email"
	uup := &models.UpdateableProperties{
		Email: &email,
	}
	suite.mockCache.On("ExpireUserProperties", mock.Anything, Users[AdminUserID]).Return(nil)
	suite.mockBackend.On("UpdateProperties", mock.Anything, uup, Users[AdminUserID]).Return(nil)

	err := suite.backend.UpdateProperties(context.Background(), uup, Users[AdminUserID])
	assert.NoError(suite.T(), err)
	suite.mockBackend.AssertExpectations(suite.T())
}

func (suite *BackendTest) TestSoftDeleteUser() {
	suite.mockBackend.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: true}).Return(Users[AdminUserID], nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, []models.Properties{*Users[AdminUserID]}, true).Return(nil)
	suite.mockBackend.On("SoftDeleteUser", mock.Anything, AdminUserID).Return(nil)

	err := suite.backend.SoftDeleteUser(context.Background(), AdminUserID)
	assert.NoError(suite.T(), err)
	suite.mockBackend.AssertExpectations(suite.T())
}

func (suite *BackendTest) TestHardDeleteUser() {
	suite.mockBackend.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: true}).Return(Users[AdminUserID], nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, []models.Properties{*Users[AdminUserID]}, true).Return(nil)
	suite.mockBackend.On("HardDeleteUser", mock.Anything, AdminUserID, true).Return(Users[AdminUserID], nil)

	prop, err := suite.backend.HardDeleteUser(context.Background(), AdminUserID, true)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), prop, Users[AdminUserID])
	suite.mockBackend.AssertExpectations(suite.T())
}

func (suite *BackendTest) TestUnDeleteUser() {
	suite.mockBackend.On("GetUserPropertiesByID", mock.Anything, AdminUserID, util.ReadOptions{ReadFromMaster: true}).Return(Users[AdminUserID], nil)
	suite.mockCache.On("CacheUsersProperties", mock.Anything, []models.Properties{*Users[AdminUserID]}, true).Return(nil)
	suite.mockBackend.On("UndeleteUser", mock.Anything, AdminUserID).Return(nil)

	err := suite.backend.UndeleteUser(context.Background(), AdminUserID)
	assert.NoError(suite.T(), err)
	suite.mockBackend.AssertExpectations(suite.T())
}

func TestGetSuite(t *testing.T) {
	suite.Run(t, new(BackendTest))
}
