package db

import (
	"errors"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/models"
)

const (
	sqlSoftDeleteUser = `UPDATE "users" SET` +
		` "deleted_on" = LOCALTIMESTAMP` +
		` WHERE "id" = $1`

	sqlHardDeleteUser           = `DELETE FROM "users" WHERE "id" = $1`
	sqlHardDeleteUserEmail      = `DELETE FROM "user_email_properties" WHERE "user_id" = $1`
	sqlHardDeleteUserImages     = `DELETE FROM "user_image_properties" WHERE "user_id" = $1`
	sqlHardDeleteUserModeration = `DELETE FROM "user_moderation_properties" WHERE "user_id" = $1`
	sqlHardDeleteUserAd         = `DELETE FROM "user_ad_properties" WHERE "user_id" = $1`
	sqlHardDeleteUserBroadcast  = `DELETE FROM "user_broadcast_properties" WHERE "user_id" = $1`
	sqlHardDeleteUserChannel    = `DELETE FROM "user_channel_properties" WHERE "user_id" = $1`

	sqlInsertDeletedBlockedUser = `INSERT into users (id, login, deleted_on, updated_on, created_on)` +
		` VALUES ($1, $2, $3, LOCALTIMESTAMP, LOCALTIMESTAMP)`

	sqlUndeleteUser = `UPDATE "users" SET "deleted_on" = null WHERE "id" = $1`
)

// HardDeleteUser purges user data. Authorization of this action should occur
// before calling this method.
func (c *siteDBImpl) HardDeleteUser(ctx context.Context, ID string, skipBlock bool) (*models.Properties, error) {
	props, err := c.GetUserPropertiesByID(ctx, ID, util.ReadOptions{ReadFromMaster: true})
	if err != nil {
		return nil, err
	}

	var login string
	if props.Login != nil {
		login = *props.Login
	}

	// Get images before user is deleted
	images, err := c.GetUserImages(ctx, ID, login)
	if err != nil {
		return nil, err
	}

	if err := c.hardDeleteUserTables(ctx, ID, login, skipBlock); err != nil {
		return nil, err
	}

	// Delete images
	imagePaths, _ := massageImageProperties(ctx, images, login, nil)

	if len(imagePaths) != 0 {
		err = c.s3.BatchDelete(imagePaths)
		if err != nil {
			return nil, err
		}
	}

	return props, err
}

func (c *siteDBImpl) hardDeleteUserTables(ctx context.Context, ID, login string, skipBlock bool) error {
	// Create transaction
	tx, err := backend.OpenTx(ctx, c.mdb, "hard delete user")
	if err != nil {
		return err
	}
	defer func() {
		tx.Close(ctx, err)
	}()

	// Delete user and associated tables
	var tables = []struct {
		name string
		sql  string
	}{
		{
			name: "user_hard_delete_users",
			sql:  sqlHardDeleteUser,
		}, {
			name: "user_hard_delete_email",
			sql:  sqlHardDeleteUserEmail,
		}, {
			name: "user_hard_delete_images",
			sql:  sqlHardDeleteUserImages,
		}, {
			name: "user_hard_delete_moderation",
			sql:  sqlHardDeleteUserModeration,
		},
		{
			name: "user_hard_delete_ad",
			sql:  sqlHardDeleteUserAd,
		},
		{
			name: "user_hard_delete_broadcast",
			sql:  sqlHardDeleteUserBroadcast,
		},
		{
			name: "user_hard_delete_channel",
			sql:  sqlHardDeleteUserChannel,
		},
	}

	for _, table := range tables {
		_, err = tx.Exec(
			ctx, table.name,
			table.sql,
			ID,
		)

		if err != nil {
			return err
		}
	}

	if !skipBlock {
		// Insert block record
		_, err = tx.Exec(
			ctx,
			"user_hard_delete_block_insert",
			sqlInsertDeletedBlockedUser,
			ID,
			login,
			time.Now())
	}

	return err
}

// SoftDeleteUser marks as user as deleted while keeping their data.
func (c *siteDBImpl) SoftDeleteUser(ctx context.Context, ID string) error {
	result, err := c.mdb.Exec(ctx, "soft_delete_user", sqlSoftDeleteUser, ID)
	if err != nil {
		return errx.New(err)
	}
	return errx.New(ensureAffected(result, 1, "failed to mark user as deleted"))
}

func (c *siteDBImpl) UndeleteUser(ctx context.Context, ID string) error {
	result, err := c.mdb.Exec(ctx, "undelete_user", sqlUndeleteUser, ID)
	if err != nil {
		return errx.New(err)
	}
	return errx.New(ensureAffected(result, 1, "failed to mark user as undeleted"))
}

func ensureAffected(result db.Result, expected int64, errMsg string) error {
	affected, err := result.RowsAffected()
	if err != nil {
		return err
	}

	if affected != affected {
		return errors.New(errMsg)
	}
	return nil
}
