package db

import (
	"testing"

	"github.com/stretchr/testify/mock"

	"golang.org/x/net/context"

	"code.justin.tv/web/users-service/database/mocks"
	s3mocks "code.justin.tv/web/users-service/internal/clients/s3/mocks"
	"code.justin.tv/web/users-service/internal/image"
)

func TestSoftDeleteUser(t *testing.T) {
	userID := "1"
	ctx := context.Background()

	result := &mocks.Result{}
	result.On("RowsAffected").Return(int64(1), nil)
	defer result.AssertExpectations(t)

	mdb := &mocks.Querier{}
	mdb.On("Exec", ctx, "soft_delete_user", mock.Anything, []interface{}{userID}).Return(result, nil)
	defer mdb.AssertExpectations(t)

	ub := &siteDBImpl{mdb: mdb}

	if err := ub.SoftDeleteUser(ctx, userID); err != nil {
		t.Fatal(err)
	}
}

func TestUndeleteUser(t *testing.T) {
	userID := "1"
	ctx := context.Background()

	result := &mocks.Result{}
	result.On("RowsAffected").Return(int64(1), nil)
	defer result.AssertExpectations(t)

	mdb := &mocks.Querier{}
	mdb.On("Exec", ctx, "undelete_user", mock.Anything, []interface{}{userID}).Return(result, nil)
	defer mdb.AssertExpectations(t)

	ub := &siteDBImpl{mdb: mdb}

	if err := ub.UndeleteUser(ctx, userID); err != nil {
		t.Fatal(err)
	}
}

func TestHardDeleteUser(t *testing.T) {
	userID := "1"
	ctx := context.Background()

	mdb := &mocks.Querier{}
	defer mdb.AssertExpectations(t)

	// GetUserPropertiesByID
	userRow := &mocks.Row{}
	defer userRow.AssertExpectations(t)
	userRow.On("Scan", mock.Anything).Return(nil)
	mdb.On("QueryRow", ctx, "user_properties_by_u.id", mock.Anything, mock.Anything).Return(userRow)

	imageRow := &mocks.Row{}
	defer imageRow.AssertExpectations(t)
	imageRow.On("Scan", mock.Anything).Return(nil)

	mdb.On("QueryRow", ctx, "user_images", mock.Anything, mock.Anything).Return(imageRow)

	// Deletes
	tx := &mocks.Tx{}
	defer tx.AssertExpectations(t)

	mdb.On("Begin", ctx, mock.Anything, mock.Anything).Return(tx, nil)

	tx.On("Exec", ctx, "user_hard_delete_block_insert", mock.Anything, mock.Anything).Return(nil, nil)
	tx.On("Exec", ctx, mock.Anything, mock.Anything, []interface{}{userID}).Return(nil, nil).Times(7)
	tx.On("Commit").Return(nil)

	err := image.SetUpImageReader()
	if err != nil {
		t.Fatal(err)
	}

	s3 := &s3mocks.S3Client{}
	s3.On("BatchDelete", mock.Anything).Return(nil)

	ub := &siteDBImpl{mdb: mdb, s3: s3}

	_, err = ub.HardDeleteUser(ctx, userID, false)
	if err != nil {
		t.Fatal(err)
	}
}
