package db

import (
	"fmt"
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/database"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	sqlSelectGlobaPrivilegedUsers = `SELECT users.login, user_role_properties.admin, user_role_properties.subadmin, user_role_properties.global_mod ` +
		`FROM users, user_role_properties ` +
		`WHERE users.id = user_role_properties.user_id`
)

func (c *siteDBImpl) GetGlobalPrivilegedUsers(ctx context.Context, roles []string) (*models.GlobalPrivilegedUsers, error) {
	roleQuery := BuildRoleQuery(roles)
	rows, err := c.sdb.Query(ctx, "global_privileged_users", roleQuery)
	if err != nil {
		return nil, errx.New(err)
	}
	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbResults, err := GetGlobalUsersFromRows(rows, roles)

	if err != nil {
		return nil, err
	}

	return dbResults, errx.New(err)

}

func GetGlobalUsersFromRows(rows database.Rows, roles []string) (dbresult *models.GlobalPrivilegedUsers, err error) {

	var username string
	var isAdmin, isSubAdmin, isGlobalMod bool
	var admins, subAdmins, globalMods []string

	for rows.Next() {
		err = rows.Scan(&username, &isAdmin, &isSubAdmin, &isGlobalMod)

		if err != nil {
			return nil, err
		}

		if isAdmin && (roles == nil || contains(roles, "admin")) {
			admins = append(admins, username)
		}
		if isSubAdmin && (roles == nil || contains(roles, "subadmin")) {
			subAdmins = append(subAdmins, username)
		}
		if isGlobalMod && (roles == nil || contains(roles, "global_mod")) {
			globalMods = append(globalMods, username)
		}
	}

	dbresult = &models.GlobalPrivilegedUsers{
		Admins:     admins,
		SubAdmins:  subAdmins,
		GlobalMods: globalMods,
	}

	return dbresult, nil
}

func BuildRoleQuery(roles []string) string {
	if len(roles) == 0 {
		return sqlSelectGlobaPrivilegedUsers
	}
	var roleSuffix []string
	var querySuffix string

	for _, role := range roles {
		roleSuffix = append(roleSuffix, fmt.Sprintf("%s = true", role))
	}
	querySuffix = fmt.Sprintf("and (%s)", strings.Join(roleSuffix, " or "))
	query := strings.Join([]string{sqlSelectGlobaPrivilegedUsers, querySuffix}, " ")
	return query
}

func contains(roles []string, role string) bool {
	for _, r := range roles {
		if r == role {
			return true
		}
	}
	return false
}
