package db

import (
	"database/sql"
	"fmt"
	"strings"

	cdb "code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/errx"
	. "code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	sqlSelectLogins = `SELECT` +
		` u.id,` +
		` login,` +
		` renamed_login_block_record` +
		` FROM users u` +
		` WHERE`
)

func (c *siteDBImpl) GetLogins(ctx context.Context, logins []string) ([]models.LoginProperties, error) {
	var querySuffix string
	var queryParams []interface{}
	var idSuffix []string

	for i, id := range logins {
		idSuffix = append(idSuffix, fmt.Sprintf("$%d", i+1))
		queryParams = append(queryParams, id)
	}
	querySuffix = fmt.Sprintf("login IN (%s)", strings.Join(idSuffix, ","))

	query := strings.Join([]string{sqlSelectLogins, querySuffix}, " ")
	userQuery := UserQuery{sql: query, params: queryParams}

	rows, err := c.sdb.Query(ctx, "get_logins", userQuery.sql, userQuery.params...)
	if err != nil {
		return nil, errx.New(err, errx.Fields{"num_logins": len(logins)})
	}
	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbResults, err := GetLoginsFromRows(rows)
	if err != nil {
		return nil, err
	}

	return dbResults, errx.New(err)
}

func GetLoginsFromRow(row cdb.Row) (*models.LoginProperties, error) {
	up := models.LoginProperties{}

	err := row.Scan(
		&up.ID,
		&up.Login,
		&up.RenamedLoginBlockRecord,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, errx.New(ErrNoProperties)
	case err != nil:
		return nil, errx.New(err)
	default:
	}

	return &up, nil
}

func GetLoginsFromRows(rows database.Rows) ([]models.LoginProperties, error) {
	results := []models.LoginProperties{}

	for rows.Next() {
		up, err := GetLoginsFromRow(rows)
		if err != nil {
			return nil, err
		}
		results = append(results, *up)
	}

	err := rows.Err()
	if err != nil {
		return nil, errx.New(err)
	}

	return results, nil
}
