package db

import (
	"database/sql"

	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/web/users-service/internal/image"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	sqlSelectUserImages = `SELECT` +
		` u.id,` +
		` profile_banner,` +
		` channel_offline_image,` +
		` u.profile_image` +
		` FROM users u` +
		` LEFT JOIN user_image_properties ui on (u.id = ui.user_id)` +
		` WHERE u.id = $1`
)

func (c *siteDBImpl) GetUserImages(ctx context.Context, id string, login string) (*models.ImageProperties, error) {
	row := c.mdb.QueryRow(ctx, "user_images", sqlSelectUserImages, id)

	up := models.ImageProperties{}

	var pbyaml *string
	var coyaml *string
	var upiyaml *string
	err := row.Scan(
		&up.ID,
		&pbyaml,
		&coyaml,
		&upiyaml,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errx.New(err)
	default:
	}

	up.ProfileBanner, up.DefaultProfileBanner, err = image.GetProfileBanners(pbyaml, login)
	if err != nil {
		if metricErr := config.ClusterStatsd().Inc("parse.profile_banner.failure", 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
		}
	}

	up.ChannelOfflineImage, up.DefaultChannelOfflineImage, err = image.GetChannelOfflineImages(coyaml, login)
	if err != nil {
		if metricErr := config.ClusterStatsd().Inc("parse.channel_offline_image.failure", 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
		}
	}

	up.ProfileImage, up.DefaultProfileImage, err = image.GetProfileImages(upiyaml, login, id)
	if err != nil {
		if metricErr := config.ClusterStatsd().Inc("parse.profile_image.failure", 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
		}
	}

	return &up, err
}
