package db

import (
	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

func (c *siteDBImpl) GetUserPropertiesByID(ctx context.Context, idString string, opts util.ReadOptions) (*models.Properties, error) {
	return c.GetUserProperties(ctx, "u.id", idString, opts)
}

func (c *siteDBImpl) GetUserPropertiesByLogin(ctx context.Context, login string, opts util.ReadOptions) (*models.Properties, error) {
	return c.GetUserProperties(ctx, "login", login, opts)
}

func (c *siteDBImpl) GetUserPropertiesByDisplayName(ctx context.Context, idString string, opts util.ReadOptions) (*models.Properties, error) {
	return c.GetUserProperties(ctx, "displayname", idString, opts)
}

func (c *siteDBImpl) GetUserProperties(ctx context.Context, field string, identifier string, opts util.ReadOptions) (dbResults *models.Properties, err error) {
	var db database.Querier
	if opts.ReadFromMaster {
		db = c.mdb
	} else {
		db = c.sdb
	}
	userQuery := BuildQuery(field, []string{identifier})
	row := db.QueryRow(ctx, fmt.Sprintf("user_properties_by_%v", field), userQuery.sql, userQuery.params...)
	return GetUserPropertiesFromRow(row)
}

func (c *siteDBImpl) GetUsersProperties(ctx context.Context, field string, identifiers []string) (dbResults []models.Properties, err error) {
	userQuery := BuildQuery(field, identifiers)
	rows, err := c.sdb.Query(ctx, "user_properties_bulk", userQuery.sql, userQuery.params...)
	if err != nil {
		return nil, errx.New(err, errx.Fields{
			"field": field,
			"count": len(identifiers),
		})
	}
	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbResults, err = GetUserPropertiesFromRows(rows)
	if err != nil {
		return nil, errx.New(err, errx.Fields{
			"field": field,
			"count": len(identifiers),
		})
	}

	return dbResults, errx.New(err, errx.Fields{
		"field": field,
		"count": len(identifiers),
	})
}

func (c *siteDBImpl) GetUserPropertiesBulk(ctx context.Context, params *models.FilterParams) ([]models.Properties, error) {
	upsc := make(chan []models.Properties)
	errc := make(chan error)

	fis := []util.Fieldidentifier{
		{Field: "u.id", Ids: params.IDs},
		{Field: "login", Ids: params.Logins},
		{Field: "lower(email)", Ids: params.Emails},
		{Field: "lower(displayname)", Ids: params.DisplayNames},
		{Field: "remote_ip", Ids: params.Ips},
	}

	wait := 0
	for _, fi := range fis {
		if len(fi.Ids) > 0 {
			wait++
			go func(fi util.Fieldidentifier) {
				ups, err := c.GetUsersProperties(ctx, fi.Field, fi.Ids)
				if err != nil {
					errc <- err
					return
				}
				upsc <- ups
			}(fi)
		}
	}

	ups := []models.Properties{}
	for i := 0; i < wait; i++ {
		select {
		case upsres := <-upsc:
			if upsres != nil {
				ups = append(ups, upsres...)
			}
		case err := <-errc:
			return nil, err
		}
	}
	return ups, nil
}

func (c *siteDBImpl) GetUserPropertiesLike(ctx context.Context, field string, pattern string) (dbResults []models.Properties, err error) {
	userQuery := BuildQueryLike(field, pattern)
	rows, err := c.sdb.Query(ctx, "user_properties_like", userQuery.sql, userQuery.params...)
	if err != nil {
		return nil, errx.New(err)
	}
	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbResults, err = GetUserPropertiesFromRows(rows)
	return dbResults, errx.New(err)
}
