package db

import (
	"database/sql"
	"errors"
	"testing"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database/mocks"
	. "github.com/smartystreets/goconvey/convey"
	. "github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

func TestStart(t *testing.T) {
	Convey("When calling GetUserProperties", t, func() {
		sdb := &mocks.Querier{}
		mdb := &mocks.Querier{}
		backend := New(mdb, sdb, nil)
		ctx := context.Background()
		Convey("with only a login name", func() {
			login := []string{"dadbot"}
			userQuery := BuildQuery("login", login)

			Convey("and the DB returns a valid row", func() {
				rows := &mocks.Rows{}

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "login", login)
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})

			Convey("and the DB returns no row", func() {
				rows := &mocks.Rows{}

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(sql.ErrNoRows)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "login", login)
				So(properties, ShouldBeEmpty)
				So(errx.Unwrap(err), ShouldHaveSameTypeAs, util.ErrNoProperties)
			})

			Convey("and the DB returns an error", func() {
				rows := &mocks.Rows{}
				dbErr := errors.New("this should never happen")

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(dbErr)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "login", login)
				So(properties, ShouldBeEmpty)
				So(errx.Unwrap(err), ShouldEqual, dbErr)
			})
		})

		Convey("with only an email", func() {
			emails := []string{"dadbot@yahoo.com"}
			userQuery := BuildQuery("lower(email)", emails)
			Convey("and the DB returns a valid row", func() {
				rows := &mocks.Rows{}

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "lower(email)", emails)
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})

			Convey("and the DB returns no row", func() {
				rows := &mocks.Rows{}

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(sql.ErrNoRows)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "lower(email)", emails)
				So(properties, ShouldBeEmpty)
				So(errx.Unwrap(err), ShouldHaveSameTypeAs, util.ErrNoProperties)
			})

			Convey("and the DB returns an error", func() {
				rows := &mocks.Rows{}
				dbErr := errors.New("this should never happen")

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(dbErr)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "lower(email)", emails)
				So(properties, ShouldBeNil)
				So(errx.Unwrap(err), ShouldEqual, dbErr)
			})
		})

		Convey("with only an id", func() {
			idS := []string{"5"}
			userQuery := BuildQuery("u.id", idS)
			Convey("and the DB returns a valid row", func() {
				rows := &mocks.Rows{}
				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "u.id", idS)
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})

			Convey("and the DB returns no row", func() {
				rows := &mocks.Rows{}

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(sql.ErrNoRows)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "u.id", idS)
				So(properties, ShouldBeEmpty)
				So(errx.Unwrap(err), ShouldHaveSameTypeAs, util.ErrNoProperties)
			})

			Convey("and the DB returns an error", func() {
				rows := &mocks.Rows{}
				dbErr := errors.New("this should never happen")

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(dbErr)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "u.id", idS)
				So(properties, ShouldBeNil)
				So(errx.Unwrap(err), ShouldEqual, dbErr)
			})
		})

		Convey("with multiple login names", func() {
			logins := []string{"dadbot", "creative"}

			Convey("and the DB returns valid rows", func() {
				rows := &mocks.Rows{}
				rows.On("Next", Anything).Return(true).Twice()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				userQuery := BuildQuery("login", logins)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "login", logins)
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("with multiple login emails", func() {
			emails := []string{"dadbot@yahoo.com", "dadbot@gmail.com"}

			Convey("and the DB returns valid rows", func() {
				rows := &mocks.Rows{}
				rows.On("Next", Anything).Return(true).Twice()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				userQuery := BuildQuery("lower(email)", emails)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "lower(email)", emails)
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("with multiple IDs", func() {
			idStrings := []string{"5", "256"}
			Convey("and the DB returns valid rows", func() {
				rows := &mocks.Rows{}
				rows.On("Next", Anything).Return(true).Twice()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				userQuery := BuildQuery("u.id", idStrings)

				sdb.On("Query", ctx, "user_properties_bulk", userQuery.sql, userQuery.params).Return(rows, nil)

				properties, err := backend.GetUsersProperties(ctx, "u.id", idStrings)
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}
