package db

import (
	"database/sql"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/models"
)

var (
	sqlUpdateUsersPhoneNumberProperties = `UPDATE user_phone_number_properties set phone_number = $2,` +
		` phone_number_verified = false,` +
		` phone_number_verification_code = $3,` +
		` phone_number_verification_sent_at = LOCALTIMESTAMP,` +
		` updated_at = LOCALTIMESTAMP` +
		` where user_id = $1`
	sqlInsertUsersPhoneNumber = `INSERT INTO user_phone_number_properties` +
		` (user_id,phone_number,phone_number_verified,phone_number_verification_code,phone_number_verification_sent_at,created_at,updated_at)` +
		` VALUES ($1,$2,false,$3,LOCALTIMESTAMP,LOCALTIMESTAMP,LOCALTIMESTAMP)`
	sqlVerifyUsersPhoneNumber = `UPDATE user_phone_number_properties set phone_number_verified = true,` +
		` phone_number_verification_code = null,` +
		` phone_number_verification_sent_at = null,` +
		` updated_at = LOCALTIMESTAMP` +
		` where user_id = $1`
	sqlSelectPhoneNumberVerifyCode = `SELECT phone_number, phone_number_verification_code, phone_number_verification_sent_at` +
		` FROM user_phone_number_properties WHERE user_id = $1`
)

func (c *siteDBImpl) setUserPhoneNumber(ctx context.Context, id, number, code string) error {
	result, err := c.mdb.Exec(ctx, "user_phone_number_update", sqlUpdateUsersPhoneNumberProperties,
		id, number, code)
	if err != nil {
		return errx.New(err)
	}

	affected, err := result.RowsAffected()
	if err != nil {
		return errx.New(err)
	}
	if affected == 0 {
		_, err = c.mdb.Exec(ctx, "user_phone_number_insert", sqlInsertUsersPhoneNumber,
			id, number, code)
	}

	if err != nil {
		return errx.New(err)
	}

	return c.bumpUserUpdatedOn(ctx, id)
}

func (c *siteDBImpl) GetUserPhoneNumber(ctx context.Context, id string) (*models.PhoneNumberProperties, error) {
	props := &models.PhoneNumberProperties{}
	row := c.mdb.QueryRow(ctx, "user_phone_number_select", sqlSelectPhoneNumberVerifyCode,
		id)

	var tempPN, tempCode *string
	var tempTime *time.Time
	err := row.Scan(&tempPN, &tempCode, &tempTime)
	if err == sql.ErrNoRows {
		return props, nil
	} else if err != nil {
		return nil, err
	}
	if tempPN != nil {
		props.PhoneNumber = *tempPN
	}
	if tempCode != nil {
		props.Code = *tempCode
	}
	if tempTime != nil {
		props.SentAt = *tempTime
	}
	return props, errx.New(err)
}

func (c *siteDBImpl) VerifyUserPhoneNumber(ctx context.Context, id string) error {
	_, err := c.mdb.Exec(ctx, "user_phone_number_verified", sqlVerifyUsersPhoneNumber, id)
	if err != nil {
		return errx.New(err)
	}

	return c.bumpUserUpdatedOn(ctx, id)
}
