package db

import (
	"database/sql"
	"fmt"
	"strings"

	cdb "code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	. "code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database"
	"code.justin.tv/web/users-service/internal/image"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	sqlSelectPropertiesWherePartial = `SELECT` +
		` u.id,` +
		` login,` +
		` birthday,` +
		` dmca_violation,` +
		` terms_of_service_violation,` +
		` deleted_on,` +
		` language,` +
		` category,` +
		` remote_ip,` +
		` email,` +
		` last_login,` +
		` banned_until,` +
		` dmca_violation_count,` +
		` tos_violation_count,` +
		` admin,` +
		` subadmin,` +
		` global_mod,` +
		` updated_on,` +
		` created_on,` +
		` COALESCE(displayname, login) as displayname,` +
		` description,` +
		` profile_image,` +
		` location,` +
		` uep.email_verified,` +
		` upnp.phone_number,` +
		` last_login_change_date` +
		` FROM users u` +
		` LEFT JOIN user_moderation_properties ump on (u.id = ump.user_id)` +
		` LEFT JOIN user_role_properties urp on (u.id = urp.user_id)` +
		` LEFT JOIN user_email_properties uep on (u.id = uep.user_id)` +
		` LEFT JOIN user_phone_number_properties upnp on (u.id = upnp.user_id)` +
		` WHERE`
)

type UserQuery struct {
	sql    string
	params []interface{}
}

func BuildQuery(field string, identifiers []string) UserQuery {
	var querySuffix string
	var queryParams []interface{}

	prefix, postfix := "", ""
	if field == "displayname" {
		prefix, postfix = "LOWER(", ")"
	}

	if len(identifiers) > 0 {

		var idSuffix []string

		for i, id := range identifiers {
			idSuffix = append(idSuffix, fmt.Sprintf("%s$%d%s", prefix, i+1, postfix))
			queryParams = append(queryParams, id)
		}
		querySuffix = fmt.Sprintf("%s%s%s IN (%s)", prefix, field, postfix, strings.Join(idSuffix, ","))
	}

	query := strings.Join([]string{sqlSelectPropertiesWherePartial, querySuffix}, " ")
	return UserQuery{sql: query, params: queryParams}
}

func BuildQueryLike(field string, pattern string) UserQuery {
	var querySuffix string
	var queryParams []interface{}

	prefix, postfix := "LOWER(", ")"
	querySuffix = fmt.Sprintf("%s%s%s LIKE $1 LIMIT 10", prefix, field, postfix)
	queryParams = append(queryParams, pattern)

	query := strings.Join([]string{sqlSelectPropertiesWherePartial, querySuffix}, " ")
	return UserQuery{sql: query, params: queryParams}
}

func GetUserPropertiesFromRow(row cdb.Row) (*models.Properties, error) {
	var piyaml *string
	up := models.Properties{}

	err := row.Scan(
		&up.ID,
		&up.Login,
		&up.Birthday,
		&up.DmcaViolation,
		&up.TermsOfServiceViolation,
		&up.DeletedOn,
		&up.Language,
		&up.Category,
		&up.RemoteIP,
		&up.Email,
		&up.LastLogin,
		&up.BannedUntil,
		&up.DmcaViolationCount,
		&up.TosViolationCount,
		&up.Admin,
		&up.Subadmin,
		&up.GlobalMod,
		&up.UpdatedOn,
		&up.CreatedOn,
		&up.Displayname,
		&up.Description,
		&piyaml,
		&up.Location,
		&up.EmailVerified,
		&up.PhoneNumber,
		&up.LastLoginChangeDate,
	)

	switch {
	case err == sql.ErrNoRows:
		return nil, errx.New(ErrNoProperties)
	case err != nil:
		return nil, errx.New(err)
	default:
	}

	if up.Login != nil {
		up.ProfileImage, _, err = image.GetProfileImages(piyaml, *up.Login, up.ID)
		if err != nil {
			if metricErr := config.ClusterStatsd().Inc("parse.profile_image.failure", 1, 0.1); metricErr != nil {
				logx.Warn(context.Background(), fmt.Sprintf("error incrementing image metrics: %s", metricErr))
			}
		}
	}

	return &up, nil
}

func GetUserPropertiesFromRows(rows database.Rows) ([]models.Properties, error) {
	results := []models.Properties{}

	for rows.Next() {
		up, err := GetUserPropertiesFromRow(rows)
		if err != nil {
			return nil, err
		}
		results = append(results, *up)
	}

	err := rows.Err()
	if err != nil {
		return nil, errx.New(err)
	}

	return results, nil
}
