package db

import (
	"errors"

	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/common/yimg"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	SqlUpdateUserImageProperties = `UPDATE "user_image_properties" SET` +
		` "channel_offline_image" = $2,` +
		` "profile_banner" = $3,` +
		` "profile_image" = $4,` +
		` "updated_at" = LOCALTIMESTAMP` +
		` WHERE "user_id" = $1`

	SqlInsertUserImageProperties = `INSERT into user_image_properties (user_id, channel_offline_image, profile_banner, profile_image, created_at, updated_at)` +
		` VALUES ($1, $2, $3, $4, LOCALTIMESTAMP, LOCALTIMESTAMP)`

	SqlUpdateProfileImage = `UPDATE "users" SET` +
		` "profile_image" = $2,` +
		` "updated_on" = LOCALTIMESTAMP` +
		` WHERE "id" = $1`
)

func (c *siteDBImpl) SetUserImageProperties(ctx context.Context, updates models.ImageProperties) error {
	tx, err := backend.OpenTx(ctx, c.mdb, "set_user_image_properties")
	if err != nil {
		return err
	}
	defer func() {
		tx.Close(ctx, err)
	}()

	var cImage *string
	var pImage *string
	var pbImage *string

	if updates.DefaultChannelOfflineImage != nil {
		cImage = updates.DefaultChannelOfflineImage
	} else if image, err := yimg.ChannelOfflineImagesToString(updates.ChannelOfflineImage); err == nil {
		cImage = image
	} else {
		if metricErr := config.ClusterStatsd().Inc("parse.channel_offline_image.failure", 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
		}
	}

	if updates.DefaultProfileImage != nil {
		pImage = updates.DefaultProfileImage
	} else if image, err := yimg.ProfileImagesToString(updates.ProfileImage); err == nil {
		pImage = image
	} else {
		if metricErr := config.ClusterStatsd().Inc("parse.profile_image.failure", 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
		}
	}

	if updates.DefaultProfileBanner != nil {
		pbImage = updates.DefaultProfileBanner
	} else if image, err := yimg.ProfileBannersToString(updates.ProfileBanner); err == nil {
		pbImage = image
	} else {
		if metricErr := config.ClusterStatsd().Inc("parse.profile_banner.failure", 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
		}
	}

	result, err := tx.Exec(
		ctx,
		"user_image_properties_update",
		SqlUpdateUserImageProperties,
		updates.ID,
		cImage,
		pbImage,
		pImage,
	)

	if err != nil {
		return errx.New(errors.New("Failed to exec set_user_image_properties"), errx.Fields{util.DBError: err})
	}

	affected, err := result.RowsAffected()
	if err != nil {
		return errx.New(errors.New("Failed to exec set_user_image_properties"), errx.Fields{util.DBError: err})
	}

	if affected == 0 {
		_, err := tx.Exec(
			ctx,
			"user_image_properties_insert",
			SqlInsertUserImageProperties,
			updates.ID,
			cImage,
			pbImage,
			pImage,
		)

		if err != nil {
			return errx.New(errors.New("Failed to exec set_user_image_properties"), errx.Fields{util.DBError: err})
		}
	}

	_, err = tx.Exec(
		ctx,
		"user_image_properties_update",
		SqlUpdateProfileImage,
		updates.ID,
		pImage,
	)

	if err != nil {
		return errx.New(errors.New("Failed to exec set_user_image_properties"), errx.Fields{util.DBError: err})
	}

	return nil
}
