package db

import (
	"fmt"
	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/common/yimg"
	"code.justin.tv/web/users-service/backend"
	. "code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/configs"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	sqlUpdateUsersProperties = `UPDATE "users" SET` +
		` "email" = $2,` +
		` "displayname" = $3,` +
		` "description" = $4,` +
		` "language" = $5,` +
		` "updated_on" = $6,` +
		` "last_login" = $7,` +
		` "remote_ip" = $8,` +
		` "location" = $9` +
		` WHERE "id" = $1`
	sqlUpdateUsersLoginProperties = `UPDATE "users" SET` +
		` "email" = $2,` +
		` "displayname" = $3,` +
		` "description" = $4,` +
		` "language" = $5,` +
		` "updated_on" = $6,` +
		` "login" = $7,` +
		` "last_login" = $8,` +
		` "remote_ip" = $9,` +
		` "location" = $10,` +
		` "last_login_change_date" = (now())` +
		` WHERE "id" = $1`

	sqlUpdateEmailProperties = `WITH UPSERT` +
		` AS (` +
		` UPDATE user_email_properties` +
		` SET email_verified = $2 ` +
		` WHERE USER_ID = $1` +
		` RETURNING *` +
		` )` +
		` INSERT into user_email_properties (user_id, email_verified, created_at, updated_at)` +
		` SELECT $1, $2, LOCALTIMESTAMP, LOCALTIMESTAMP` +
		` WHERE NOT EXISTS (SELECT * FROM upsert)`

	sqlUpdateUserModerationProperties = `UPDATE "user_moderation_properties" SET ` +
		` dmca_violation_count = dmca_violation_count + 1,` +
		` "updated_on" = LOCALTIMESTAMP ` +
		` WHERE "user_id" = $1`
	sqlAlterDMCAViolationCount = `WITH upsert` +
		` AS (` +
		` UPDATE user_moderation_properties` +
		` SET dmca_violation_count = ` +
		` CASE WHEN dmca_violation_count + $2 < 0 THEN 0 else dmca_violation_count + $2 END,` +
		` updated_at = LOCALTIMESTAMP` +
		` WHERE user_id = $1` +
		` RETURNING *` +
		` )` +
		` INSERT INTO user_moderation_properties (user_id, banned_until, dmca_violation_count, tos_violation_count, created_at, updated_at)` +
		` SELECT` +
		` $1,` +
		` null,` +
		` CASE WHEN $2 < 0 THEN 0 ELSE 1 END,` +
		` 0,` +
		` LOCALTIMESTAMP,` +
		` LOCALTIMESTAMP` +
		` WHERE NOT EXISTS (SELECT * FROM upsert)`
	sqlBumpUsersUpdatedOn = `UPDATE "users" SET` +
		`  updated_on = LOCALTIMESTAMP` +
		` WHERE id = $1`
	sqlForceDeleteBlockedUser = `DELETE FROM "users" WHERE "login" = $1 AND "renamed_login_block_record" = TRUE`
)

func (c *siteDBImpl) bumpUserUpdatedOn(ctx context.Context, ID string) error {
	_, err := c.mdb.Exec(ctx,
		"bump_users_updated_on",
		sqlBumpUsersUpdatedOn,
		ID)

	return errx.New(err)
}

func (c *siteDBImpl) AlterDMCAStrike(ctx context.Context, ID string, delta int) error {
	_, err := c.mdb.Exec(
		ctx,
		"user_dmca_strike_alter",
		sqlAlterDMCAViolationCount,
		ID,
		delta)
	if err != nil {
		return errx.New(err)
	}

	return c.bumpUserUpdatedOn(ctx, ID)
}

func (c *siteDBImpl) UpdateProperties(ctx context.Context, uup *models.UpdateableProperties, cup *models.Properties) error {

	uup.FillFromProperties(cup)

	now := time.Now()
	var err error

	if uup.EmailVerified != nil {
		_, err = c.mdb.Exec(
			ctx,
			"user_email_properties_update",
			sqlUpdateEmailProperties,
			uup.ID,
			uup.EmailVerified,
		)
		if err != nil {
			return errx.New(err)
		}
	}

	if uup.PhoneNumber != nil {
		if err := c.setUserPhoneNumber(ctx, uup.ID, *uup.PhoneNumber, uup.PhoneNumberCode); err != nil {
			return errx.New(err)
		}
	}

	if uup.NewLogin != nil {
		if !configs.IsProduction() {
			err = c.renameLogin(ctx, *uup.Login, uup, now)
		} else {
			err = c.renameProcess(ctx, *uup.Login, uup, now)
		}
	} else {
		_, err = c.mdb.Exec(
			ctx,
			"user_uup_update",
			sqlUpdateUsersProperties,
			uup.ID,
			uup.Email,
			uup.Displayname,
			uup.Description,
			uup.Language,
			now,
			uup.LastLogin,
			uup.RemoteIP,
			uup.Location,
		)
	}

	// Add tmp extra logging to see which fields is causing rollbar errors
	if err != nil {
		logx.Error(ctx, err, logx.Fields{
			"id":          uup.ID,
			"new_login":   uup.NewLogin,
			"email":       uup.Email,
			"displayname": uup.Displayname,
			"description": uup.Description,
			"language":    uup.Language,
			"updated_on":  now,
			"last_login":  uup.LastLogin,
			"remote_ip":   uup.RemoteIP,
			"location":    uup.Location,
		})
	}

	return errx.New(err)
}

func (c *siteDBImpl) renameProcess(ctx context.Context, login string, uup *models.UpdateableProperties, now time.Time) error {
	imageProperties, err := c.GetUserImages(ctx, uup.ID, *uup.Login)
	if err != nil {
		return err
	}

	paths, fromTo := massageImageProperties(ctx, imageProperties, *uup.Login, uup.NewLogin)

	if len(fromTo) != 0 {
		err = c.s3.BatchCopy(fromTo)
		if err != nil {
			logx.Error(ctx, "failed to batch copy during rename process", logx.Fields{
				"Error": fmt.Sprintf("%v", err),
			})
		}
	}

	err = c.renameLogin(ctx, login, uup, now)
	if err != nil {
		return errx.New(err)
	}

	if len(paths) != 0 {
		err = c.s3.BatchDelete(paths)
		if err != nil {
			logx.Error(ctx, "failed to batch delete during rename process", logx.Fields{
				"Error": fmt.Sprintf("%v", err),
			})
		}
	}

	return nil
}

func massageImageProperties(ctx context.Context, imageProperties *models.ImageProperties, login string, newLoginPtr *string) (paths []string, fromTo map[string]string) {
	fromTo = make(map[string]string)
	paths = []string{}

	hasNewLogin := newLoginPtr != nil
	var newLogin string
	if hasNewLogin {
		newLogin = *newLoginPtr
	}

	if imageProperties != nil {
		if imageProperties.ProfileBanner != nil {
			for _, image := range *imageProperties.ProfileBanner {
				if image.NewFormat {
					break
				}
				origin, e := yimg.ProfileBannersName(image, login)
				if e == nil {
					paths = append(paths, origin)
					if hasNewLogin {
						newName, e := yimg.ProfileBannersName(image, newLogin)
						if e == nil {
							fromTo[origin] = newName
						}
					}
				} else {
					if metricErr := config.ClusterStatsd().Inc("parse.profile_banner.failure", 1, 0.1); metricErr != nil {
						logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
					}
				}
			}
		}

		if imageProperties.ProfileImage != nil {
			for _, image := range *imageProperties.ProfileImage {
				if image.NewFormat {
					break
				}
				origin, e := yimg.ProfileImagesName(image, login)
				if e == nil {
					paths = append(paths, origin)
					if hasNewLogin {
						newName, e := yimg.ProfileImagesName(image, newLogin)
						if e == nil {
							fromTo[origin] = newName
						}
					}
				} else {
					if metricErr := config.ClusterStatsd().Inc("parse.profile_image.failure", 1, 0.1); metricErr != nil {
						logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
					}
				}
			}
		}

		if imageProperties.ChannelOfflineImage != nil {
			for _, image := range *imageProperties.ChannelOfflineImage {
				if image.NewFormat {
					break
				}
				origin, e := yimg.ChannelImageName(image, login)
				if e == nil {
					paths = append(paths, origin)
					if hasNewLogin {
						newName, e := yimg.ChannelImageName(image, newLogin)
						if e == nil {
							fromTo[origin] = newName
						}
					}
				} else {
					if metricErr := config.ClusterStatsd().Inc("parse.channel_offline_image.failure", 1, 0.1); metricErr != nil {
						logx.Warn(ctx, fmt.Sprintf("error incrementing image metrics: %s", metricErr))
					}
				}
			}
		}
	}

	return
}

func (c *siteDBImpl) renameLogin(ctx context.Context, login string, uup *models.UpdateableProperties, now time.Time) error {
	tx, err := backend.OpenTx(ctx, c.mdb, "rename login")
	if err != nil {
		return err
	}

	defer func() {
		tx.Close(ctx, err)
	}()

	// We should be able to remove this block after all records are migrated out
	if uup.OverrideLoginBlock != nil && *uup.OverrideLoginBlock {
		_, err = tx.Exec(
			ctx,
			"user_rename_block_delete",
			sqlForceDeleteBlockedUser,
			uup.NewLogin,
		)
		if err != nil {
			return errx.New(err, errx.Fields{DBError: "Failed to exec user_rename_block_delete"})
		}
	}

	_, err = tx.Exec(
		ctx,
		"user_uulp_update",
		sqlUpdateUsersLoginProperties,
		uup.ID,
		uup.Email,
		uup.Displayname,
		uup.Description,
		uup.Language,
		now,
		uup.NewLogin,
		uup.LastLogin,
		uup.RemoteIP,
		uup.Location,
	)
	if err != nil {
		return errx.New(err, errx.Fields{DBError: "Failed to exec user_uulp_update"})
	}

	return errx.New(err, errx.Fields{DBError: "Failed to exec user_redirect_channel_update"})
}
