package db

import (
	"errors"
	"testing"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/web/users-service/database/mocks"
	"code.justin.tv/web/users-service/models"

	"code.justin.tv/web/users-service/validators"
	. "github.com/smartystreets/goconvey/convey"
)

func strPtr(str string) *string {
	return &str
}

func boolPtr(bl bool) *bool {
	return &bl
}

func TestRenameLogin(t *testing.T) {
	Convey("When calling renameLogin", t, func() {
		sdb := &mocks.Querier{}
		mdb := &mocks.Querier{}
		ub := &siteDBImpl{sdb: sdb, mdb: mdb}
		ctx := context.Background()

		Convey("with updateable properties", func() {
			tx := mocks.Tx{}

			uup := &models.UpdateableProperties{
				ID:                  "5",
				Email:               strPtr("bs@twitch.tv"),
				Displayname:         strPtr("bsbs"),
				Description:         strPtr("lolwut"),
				Language:            strPtr("en"),
				NewLogin:            strPtr("bsbsbs"),
				ReleaseDateDuration: 6 * 30 * 24 * time.Hour,
				OverrideLoginBlock:  boolPtr(true),
				LastLogin:           strPtr(time.Now().Format(validators.LastLoginFormat)),
			}
			now := time.Now()

			mockUpdateExec := func(err error) {
				tx.On("Exec", ctx, "user_uulp_update", sqlUpdateUsersLoginProperties, []interface{}{
					uup.ID,
					uup.Email,
					uup.Displayname,
					uup.Description,
					uup.Language,
					now,
					uup.NewLogin,
					uup.LastLogin,
					uup.RemoteIP,
					uup.Location,
				}).Return(&mocks.Result{}, err)
			}

			mockUpdateExecSuccess := func() {
				mockUpdateExec(nil)
			}
			mockUpdateExecFailure := func() {
				mockUpdateExec(errors.New("Garbage"))
			}

			Convey("and the DB executes sql without error", func() {
				tx.On("Commit").Return(nil)
				tx.On("Exec", ctx, "user_rename_block_delete", sqlForceDeleteBlockedUser, []interface{}{
					uup.NewLogin,
				}).Return(&mocks.Result{}, nil)
				mockUpdateExecSuccess()
				mdb.On("Begin", ctx, "rename_login_transaction").Return(&tx, nil)

				err := ub.renameLogin(ctx, "oldLogin", uup, now)
				So(err, ShouldBeNil)
			})

			Convey("and the DB executes first sql with error", func() {
				tx.On("Commit").Return(nil)
				tx.On("Rollback").Return(nil)
				tx.On("Exec", ctx, "user_rename_block_delete", sqlForceDeleteBlockedUser, []interface{}{
					uup.NewLogin,
				}).Return(&mocks.Result{}, errors.New("Garbage"))
				mockUpdateExecSuccess()
				mdb.On("Begin", ctx, "rename_login_transaction").Return(&tx, nil)

				err := ub.renameLogin(ctx, "oldLogin", uup, now)
				So(err, ShouldNotBeNil)
			})

			Convey("and the DB executes second sql with error", func() {
				tx.On("Commit").Return(nil)
				tx.On("Rollback").Return(nil)
				tx.On("Exec", ctx, "user_rename_block_delete", sqlForceDeleteBlockedUser, []interface{}{
					uup.NewLogin,
				}).Return(&mocks.Result{}, nil)
				mockUpdateExecFailure()
				mdb.On("Begin", ctx, "rename_login_transaction").Return(&tx, nil)

				err := ub.renameLogin(ctx, "oldLogin", uup, now)
				So(err, ShouldNotBeNil)
			})
		})
	})
}
