package users

import (
	"fmt"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/models"
)

func init() {
	// These values were determined by looking at the past 7 days of database rps and p99s.
	// The graphs are in the "Database" section of https://grafana.internal.justin.tv/dashboard/db/users-service-v2?refresh=30s&orgId=1&from=now-7d&to=now-1m.
	// The max concurrent requests was determined by going above the max of the past week / divided by hosts at the time.
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"Users.GetUserPropertiesByID": {
			Timeout:               5000,
			MaxConcurrentRequests: 8000,
		},
		"Users.GetUserPropertiesByLogin": {
			Timeout:               5000,
			MaxConcurrentRequests: 200,
		},
		"Users.GetUserPropertiesByDisplayname": {
			Timeout:               5000,
			MaxConcurrentRequests: 200,
		},
		// This is called by GetUserPropertiesBulk
		"Users.GetUsersProperties": {
			Timeout: 5000,
			// Called up to 5 times by GetUserProperitesBulk
			MaxConcurrentRequests: 5000 * 5,
		},
		"Users.GetUserPropertiesBulk": {
			Timeout:               5000,
			MaxConcurrentRequests: 7500,
		},
		"Users.GetUserPropertiesLike": {
			Timeout: 5000,
		},
		"Users.GetBannedUsers": {
			Timeout:               5000,
			MaxConcurrentRequests: 40,
		},
		"Users.UpdateProperties": {
			Timeout:               5000,
			MaxConcurrentRequests: 40,
		},
		"Users.UnbanUser": {
			Timeout:               5000,
			MaxConcurrentRequests: 40,
		},
		"Users.BanUser": {
			Timeout:               5000,
			MaxConcurrentRequests: 40,
		},
		"Users.AlterDMCAStrike": {
			Timeout: 5000,
		},
		"Users.GetUserBlock": {
			Timeout:               5000,
			MaxConcurrentRequests: 10,
		},
		"Users.GetUserImages": {
			Timeout:               5000,
			MaxConcurrentRequests: 600,
		},
		"Users.CreateUser": {
			Timeout:               5000,
			MaxConcurrentRequests: 30,
		},
		"Users.CreateUserEmail": {
			Timeout:               5000,
			MaxConcurrentRequests: 30,
		},
		"Users.GetUserPhoneNumber": {
			Timeout: 5000,
		},
		"Users.VerifyUserPhoneNumber": {
			Timeout: 5000,
		},
		"Users.GetGlobalPrivilegedUsers": {
			Timeout:               5000,
			MaxConcurrentRequests: 20,
		},
		"Users.GetLogins": {
			Timeout:               5000,
			MaxConcurrentRequests: 300,
		},
		"Users.SetUserImageProperties": {
			Timeout:               5000,
			MaxConcurrentRequests: 30,
		},
		"Users.HardDeleteUser": {
			Timeout:               5000,
			MaxConcurrentRequests: 10,
		},
		"Users.SoftDeleteUser": {
			Timeout:               5000,
			MaxConcurrentRequests: 10,
		},
		"Users.UndeleteUser": {
			Timeout:               5000,
			MaxConcurrentRequests: 10,
		},
	})
}

type HystrixBackend struct {
	Backend Backend
}

var _ Backend = (*HystrixBackend)(nil)

func skipError(err error) bool {
	err = errx.Unwrap(err)
	return err == util.ErrNoProperties
}

func (b *HystrixBackend) GetUserPropertiesByID(ctx context.Context, id string, opts util.ReadOptions) (*models.Properties, error) {
	var ret *models.Properties
	var err error

	hystrixErr := hystrix.Do("Users.GetUserPropertiesByID", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUserPropertiesByID circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUserPropertiesByID(ctx, id, opts)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) GetUserPropertiesByLogin(ctx context.Context, login string, opts util.ReadOptions) (*models.Properties, error) {
	var ret *models.Properties
	var err error

	hystrixErr := hystrix.Do("Users.GetUserPropertiesByLogin", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUserPropertiesByLogin circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUserPropertiesByLogin(ctx, login, opts)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) GetUserPropertiesByDisplayname(ctx context.Context, dn *string) (*models.Properties, error) {
	var ret *models.Properties
	var err error

	hystrixErr := hystrix.Do("Users.GetUserPropertiesByDisplayname", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUserPropertiesByDisplayname circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUserPropertiesByDisplayname(ctx, dn)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) GetUsersProperties(ctx context.Context, field string, identifiers []string) ([]models.Properties, error) {
	var ret []models.Properties
	var err error

	hystrixErr := hystrix.Do("Users.GetUsersProperties", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUsersProperties circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUsersProperties(ctx, field, identifiers)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) GetUserPropertiesBulk(ctx context.Context, params *models.FilterParams) ([]models.Properties, error) {
	var ret []models.Properties
	var err error

	hystrixErr := hystrix.Do("Users.GetUserPropertiesBulk", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUserPropertiesBulk circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUserPropertiesBulk(ctx, params)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) GetUserPropertiesLike(ctx context.Context, field string, pattern string) ([]models.Properties, error) {
	var ret []models.Properties
	var err error

	hystrixErr := hystrix.Do("Users.GetUserPropertiesLike", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUserPropertiesLike circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUserPropertiesLike(ctx, field, pattern)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) GetBannedUsers(ctx context.Context, until time.Time) ([]models.Properties, error) {
	var ret []models.Properties
	var err error

	hystrixErr := hystrix.Do("Users.GetBannedUsers", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetBannedUsers circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetBannedUsers(ctx, until)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) UpdateProperties(ctx context.Context, uup *models.UpdateableProperties, cup *models.Properties) error {
	var err error

	hystrixErr := hystrix.Do("Users.UpdateProperties", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.UpdateProperties circuit panic=%v", r))
			}
		}()

		err = b.Backend.UpdateProperties(ctx, uup, cup)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) UnbanUser(ctx context.Context, id string, tosCount, dmcaCount int) error {
	var err error

	hystrixErr := hystrix.Do("Users.UnbanUser", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.UnbanUser circuit panic=%v", r))
			}
		}()

		err = b.Backend.UnbanUser(ctx, id, tosCount, dmcaCount)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) BanUser(ctx context.Context, id string, warn bool, tosBan bool, dmcaStrikes int, tosStrikes int) error {
	var err error

	hystrixErr := hystrix.Do("Users.BanUser", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.BanUser circuit panic=%v", r))
			}
		}()

		err = b.Backend.BanUser(ctx, id, warn, tosBan, dmcaStrikes, tosStrikes)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) AlterDMCAStrike(ctx context.Context, id string, delta int) error {
	var err error

	hystrixErr := hystrix.Do("Users.AlterDMCAStrike", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.AlterDMCAStrike circuit panic=%v", r))
			}
		}()

		err = b.Backend.AlterDMCAStrike(ctx, id, delta)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) GetUserBlock(ctx context.Context, login string) (*models.BlockProperties, error) {
	var ret *models.BlockProperties
	var err error

	hystrixErr := hystrix.Do("Users.GetUserBlock", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUserBlock circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUserBlock(ctx, login)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) GetUserImages(ctx context.Context, id string, login string) (*models.ImageProperties, error) {
	var ret *models.ImageProperties
	var err error

	hystrixErr := hystrix.Do("Users.GetUserImages", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUserImages circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUserImages(ctx, id, login)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) CreateUser(ctx context.Context, up *models.CreateUserProperties) error {
	var err error

	hystrixErr := hystrix.Do("Users.CreateUser", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.CreateUser circuit panic=%v", r))
			}
		}()

		err = b.Backend.CreateUser(ctx, up)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) CreateUserEmail(ctx context.Context, id, code string) error {
	var err error

	hystrixErr := hystrix.Do("Users.CreateUserEmail", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.CreateUserEmail circuit panic=%v", r))
			}
		}()

		err = b.Backend.CreateUserEmail(ctx, id, code)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) GetUserPhoneNumber(ctx context.Context, id string) (*models.PhoneNumberProperties, error) {
	var ret *models.PhoneNumberProperties
	var err error

	hystrixErr := hystrix.Do("Users.GetUserPhoneNumber", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetUserPhoneNumber circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetUserPhoneNumber(ctx, id)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) VerifyUserPhoneNumber(ctx context.Context, id string) error {
	var err error

	hystrixErr := hystrix.Do("Users.VerifyUserPhoneNumber", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.VerifyUserPhoneNumber circuit panic=%v", r))
			}
		}()

		err = b.Backend.VerifyUserPhoneNumber(ctx, id)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) GetGlobalPrivilegedUsers(ctx context.Context, roles []string) (*models.GlobalPrivilegedUsers, error) {
	var ret *models.GlobalPrivilegedUsers
	var err error

	hystrixErr := hystrix.Do("Users.GetGlobalPrivilegedUsers", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetGlobalPrivilegedUsers circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetGlobalPrivilegedUsers(ctx, roles)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) GetLogins(ctx context.Context, logins []string) ([]models.LoginProperties, error) {
	var ret []models.LoginProperties
	var err error

	hystrixErr := hystrix.Do("Users.GetLogins", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.GetLogins circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetLogins(ctx, logins)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) SetUserImageProperties(ctx context.Context, updates models.ImageProperties) error {
	var err error

	hystrixErr := hystrix.Do("Users.SetUserImageProperties", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.SetUserImageProperties circuit panic=%v", r))
			}
		}()

		err = b.Backend.SetUserImageProperties(ctx, updates)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) HardDeleteUser(ctx context.Context, ID string, skipBlock bool) (*models.Properties, error) {
	var ret *models.Properties
	var err error

	hystrixErr := hystrix.Do("Users.HardDeleteUser", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.HardDeleteUser circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.HardDeleteUser(ctx, ID, skipBlock)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
func (b *HystrixBackend) SoftDeleteUser(ctx context.Context, ID string) error {
	var err error

	hystrixErr := hystrix.Do("Users.SoftDeleteUser", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.SoftDeleteUser circuit panic=%v", r))
			}
		}()

		err = b.Backend.SoftDeleteUser(ctx, ID)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
func (b *HystrixBackend) UndeleteUser(ctx context.Context, ID string) error {
	var err error

	hystrixErr := hystrix.Do("Users.UndeleteUser", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Users.UndeleteUser circuit panic=%v", r))
			}
		}()

		err = b.Backend.UndeleteUser(ctx, ID)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}
