package usersclient_external

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"

	"code.justin.tv/foundation/twitchclient"
	util "code.justin.tv/web/users-service/client"
	"code.justin.tv/web/users-service/models"
)

const (
	defaultStatSampleRate = 1.0
	defaultTimingXactName = "users_service"
	batchSize             = 100
	batchURLPath          = "/users/external"
)

// Client is an interface that exposes methods to fetch data from the users service for external service.
//go:generate mockery -name ExternalClient
type ExternalClient interface {
	GetUserByID(ctx context.Context, userID string, requester string, reqOpts *twitchclient.ReqOpts) (*models.Properties, error)
	GetUserByIDAndParams(ctx context.Context, userID string, requester string, params *models.FilterParams, reqOpts *twitchclient.ReqOpts) (*models.Properties, error)
	GetUserByLogin(ctx context.Context, login string, requester string, reqOpts *twitchclient.ReqOpts) (*models.Properties, error)
	GetUsers(ctx context.Context, requester string, params *models.FilterParams, reqOpts *twitchclient.ReqOpts) (*models.PropertiesResult, error)
	GetUsersByLoginLike(ctx context.Context, requester string, pattern string, reqOpts *twitchclient.ReqOpts) (*models.PropertiesResult, error)
	GetRenameEligibility(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*models.RenameProperties, error)
	SetUser(ctx context.Context, userID string, uup *models.UpdateableProperties, reqOpts *twitchclient.ReqOpts) error
	VerifyUserPhoneNumber(ctx context.Context, userID, code string, reqOpts *twitchclient.ReqOpts) error
	UploadUserImage(ctx context.Context, userID string, uup models.UploadableImage, reqOpts *twitchclient.ReqOpts) (*models.UploadInfo, error)
	SetUserImageMetadata(ctx context.Context, uup models.ImageProperties, reqOpts *twitchclient.ReqOpts) error
	SetUserImageMetadataAuthed(ctx context.Context, editor string, uup models.ImageProperties, reqOpts *twitchclient.ReqOpts) error
}

type clientImpl struct {
	twitchclient.Client
}

// NewClient creates a client for the users service.
func NewClient(conf twitchclient.ClientConf) (ExternalClient, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &clientImpl{twitchClient}, nil
}

func (c *clientImpl) GetUserByID(ctx context.Context, userID string, requester string, reqOpts *twitchclient.ReqOpts) (*models.Properties, error) {
	return c.GetUserByIDAndParams(ctx, userID, requester, nil, reqOpts)
}

func (c *clientImpl) GetUserByIDAndParams(ctx context.Context, userID string, requester string, params *models.FilterParams, reqOpts *twitchclient.ReqOpts) (*models.Properties, error) {
	query := url.Values{}
	util.ModifyQuery(&query, params)
	query.Add("requester", requester)

	path := (&url.URL{
		Path:     fmt.Sprintf("/users/%s/external", userID),
		RawQuery: query.Encode(),
	}).String()

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.users_service.get_user_external",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusOK:
		var decoded models.Properties
		if err := json.NewDecoder(httpResp.Body).Decode(&decoded); err != nil {
			return nil, err
		}
		return &decoded, nil

	case http.StatusNotFound:
		return nil, &util.UserNotFoundError{}

	default:
		// Unexpected result
		return nil, util.HandleUnexpectedResult(httpResp)
	}
}

func (c *clientImpl) GetUserByLogin(ctx context.Context, login string, requester string, reqOpts *twitchclient.ReqOpts) (*models.Properties, error) {
	params := &models.FilterParams{Logins: []string{login}}
	users, err := c.GetUsers(ctx, requester, params, reqOpts)
	if err != nil {
		return nil, err
	}
	if users == nil || len(users.Results) < 1 {
		return nil, &util.UserNotFoundError{}
	}
	return users.Results[0], nil
}

func (c *clientImpl) GetUsers(ctx context.Context, requester string, params *models.FilterParams, reqOpts *twitchclient.ReqOpts) (*models.PropertiesResult, error) {
	batches := util.BatchParams(params)
	ch := make(chan *util.BatchResult, len(batches))
	for _, batch := range batches {
		subCtx, cancel := context.WithCancel(ctx)
		defer cancel()
		query := url.Values{}
		query.Add("requester", requester)

		combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
			StatName: "service.users_service.get_users_batch_external",
		})

		go util.GetAsyncWithQuery(subCtx, c, query, batchURLPath, batch, ch, &combinedReqOpts)
	}

	return util.CombineBatches(ch, batches)
}

func (c *clientImpl) GetUsersByLoginLike(ctx context.Context, requester string, pattern string, reqOpts *twitchclient.ReqOpts) (*models.PropertiesResult, error) {
	query := url.Values{}
	util.ModifyQueryFieldValue(&query, "login_like", pattern)
	query.Add("requester", requester)
	path := (&url.URL{
		Path:     "/users/external",
		RawQuery: query.Encode(),
	}).String()
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.users_service.get_users_like_external",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusOK:
		var decoded models.PropertiesResult
		if err := json.NewDecoder(httpResp.Body).Decode(&decoded); err != nil {
			return nil, err
		}
		return &decoded, nil

	default:
		// Unexpected result
		return nil, util.HandleUnexpectedResult(httpResp)
	}
}

func (c *clientImpl) GetRenameEligibility(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*models.RenameProperties, error) {
	path := fmt.Sprintf("/users/%s/rename_eligible", userID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.users_service.get_rename_eligibility",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusOK:
		var decoded models.RenameProperties
		if err := json.NewDecoder(httpResp.Body).Decode(&decoded); err != nil {
			return nil, err
		}
		return &decoded, nil

	case http.StatusNotFound:
		return nil, &util.UserNotFoundError{}

	default:
		// Unexpected result
		return nil, util.HandleUnexpectedResult(httpResp)
	}
}

func (c *clientImpl) SetUser(ctx context.Context, userID string, uup *models.UpdateableProperties, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/users/%s", userID)

	bodyJson, err := json.Marshal(uup)
	if err != nil {
		return err
	}

	body := bytes.NewBuffer(bodyJson)

	req, err := c.NewRequest("PATCH", path, body)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.users_service.update_user_properties",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusNoContent:
		return nil

	case http.StatusNotFound:
		return &util.UserNotFoundError{}

	default:
		// Unexpected result
		return util.HandleUnexpectedResult(httpResp)
	}
}

func (c *clientImpl) VerifyUserPhoneNumber(ctx context.Context, userID, code string, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/users/%s/verify_phone_number", userID)

	bodyJSON, err := json.Marshal(models.PhoneNumberCodeProperties{
		Code: code,
	})
	if err != nil {
		return err
	}

	body := bytes.NewBuffer(bodyJSON)

	req, err := c.NewRequest("POST", path, body)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.users_service.verify_phone_number",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusNoContent:
		return nil

	case http.StatusNotFound:
		return &util.UserNotFoundError{}

	default:
		// Unexpected result
		return util.HandleUnexpectedResult(httpResp)
	}
}

func (c *clientImpl) SetUserImageMetadata(ctx context.Context, uup models.ImageProperties, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/users/%s/images/metadata", uup.ID)
	bodyJson, err := json.Marshal(uup)
	if err != nil {
		return err
	}

	body := bytes.NewBuffer(bodyJson)

	req, err := c.NewRequest("PATCH", path, body)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.users_service.set_image_metadata",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusNoContent:
		return nil

	default:
		// Unexpected result
		return util.HandleUnexpectedResult(httpResp)
	}
}

func (c *clientImpl) SetUserImageMetadataAuthed(ctx context.Context, editor string, uup models.ImageProperties, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/users/editor/%s/images", editor)
	bodyJson, err := json.Marshal(uup)
	if err != nil {
		return err
	}

	body := bytes.NewBuffer(bodyJson)

	req, err := c.NewRequest("PATCH", path, body)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.users_service.set_image_metadata_authed",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusNoContent:
		return nil

	default:
		// Unexpected result
		return util.HandleUnexpectedResult(httpResp)
	}
}

func (c *clientImpl) UploadUserImage(ctx context.Context, userID string, uup models.UploadableImage, reqOpts *twitchclient.ReqOpts) (*models.UploadInfo, error) {
	query := url.Values{}
	query.Add("return_upload_info_as_struct", "true")

	path := (&url.URL{
		Path:     fmt.Sprintf("/users/%s/images", userID),
		RawQuery: query.Encode(),
	}).String()

	bodyJson, err := json.Marshal(uup)
	if err != nil {
		return nil, err
	}

	body := bytes.NewBuffer(bodyJson)

	req, err := c.NewRequest("PATCH", path, body)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.users_service.upload_user_image",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusOK:
		var uploadInfo models.UploadInfo
		if err := json.NewDecoder(httpResp.Body).Decode(&uploadInfo); err != nil {
			return nil, err
		}
		return &uploadInfo, nil

	default:
		// Unexpected result
		return nil, util.HandleUnexpectedResult(httpResp)
	}
}
