package usersclient_internal

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"sync"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"

	"code.justin.tv/foundation/twitchclient"
	. "code.justin.tv/web/users-service/client"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
)

const (
	requestsMade  = "requests-made"
	requestErrors = "request-errors"
)

func TestGetUserByID(t *testing.T) {
	Convey("when calling GetUserByID", t, func() {
		ctx := context.Background()
		id := "45573555"
		Convey("when the backend is available", func() {
			Convey("with a valid user ID", func() {
				login := "wimpy"
				properties := &models.Properties{
					ID:    id,
					Login: &login,
				}

				ts := initUserTestServer(id, properties)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				returnedProperties, err := c.GetUserByID(ctx, id, nil)
				So(err, ShouldBeNil)
				So(returnedProperties, ShouldResemble, properties)
			})

			Convey("with a user ID that does not exist", func() {
				ts := initUserTestServer("69", nil)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				_, err = c.GetUserByID(ctx, id, nil)
				So(err, ShouldNotBeNil)
				So(err, ShouldHaveSameTypeAs, &UserNotFoundError{})
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetUserByID(ctx, id, nil)
			So(err, ShouldNotBeNil)
			So(returnedProperties, ShouldBeNil)
		})
	})
}

func TestGetUserByLogin(t *testing.T) {
	Convey("when calling GetUserByLogin", t, func() {
		ctx := context.Background()
		id := "45573555"
		login := "wimpy"
		query := "login=wimpy"
		expectedResult := &models.Properties{
			ID:    id,
			Login: &login,
		}
		propertiesResult := &models.PropertiesResult{
			Results: []*models.Properties{expectedResult},
		}

		Convey("when the backend is available", func() {
			Convey("with a valid login", func() {
				ts := initUsersTestServer(query, propertiesResult)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				actualResult, err := c.GetUserByLogin(ctx, login, nil)
				So(err, ShouldBeNil)
				So(actualResult, ShouldResemble, expectedResult)
			})

			Convey("with login that does not exist", func() {
				ts := initUsersTestServer(query, nil)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				_, err = c.GetUserByLogin(ctx, login, nil)
				So(err, ShouldNotBeNil)
				So(err, ShouldHaveSameTypeAs, &UserNotFoundError{})
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			actualResult, err := c.GetUserByLogin(ctx, login, nil)
			So(err, ShouldNotBeNil)
			So(actualResult, ShouldBeNil)
		})
	})
}

func TestGetUserByEmail(t *testing.T) {
	Convey("when calling GetUserByEmail", t, func() {
		ctx := context.Background()
		id := "45573555"
		email := "wimpy@twitch.tv"
		query := "email=wimpy%40twitch.tv"
		expectedResult := &models.Properties{
			ID:    id,
			Email: &email,
		}
		propertiesResult := &models.PropertiesResult{
			Results: []*models.Properties{expectedResult},
		}

		Convey("when the backend is available", func() {
			Convey("with a valid login", func() {
				ts := initUsersTestServer(query, propertiesResult)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				actualResult, err := c.GetUserByEmail(ctx, email, nil)
				So(err, ShouldBeNil)
				So(actualResult, ShouldResemble, expectedResult)
			})

			Convey("with email that does not exist", func() {
				ts := initUsersTestServer(query, nil)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				_, err = c.GetUserByEmail(ctx, email, nil)
				So(err, ShouldNotBeNil)
				So(err, ShouldHaveSameTypeAs, &UserNotFoundError{})
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			actualResult, err := c.GetUserByEmail(ctx, email, nil)
			So(err, ShouldNotBeNil)
			So(actualResult, ShouldBeNil)
		})
	})
}

func TestGetUsers(t *testing.T) {
	Convey("when calling GetUsers", t, func() {
		ctx := context.Background()

		params := &models.FilterParams{
			IDs:    []string{"45573555"},
			Logins: []string{"wimpy"},
			Emails: []string{"services@twitch.tv"},
		}

		query := "email=services%40twitch.tv&id=45573555&login=wimpy"
		propertiesResult := &models.PropertiesResult{
			Results: make([]*models.Properties, 0, 0),
		}
		for index, ID := range params.IDs {
			properties := &models.Properties{
				ID:    ID,
				Login: &params.Logins[index],
				Email: &params.Emails[index],
			}
			propertiesResult.Results = append(propertiesResult.Results, properties)
		}

		Convey("when the backend is available", func() {
			ts := initUsersTestServer(query, propertiesResult)
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetUsers(ctx, params, nil)
			So(err, ShouldBeNil)
			So(returnedProperties, ShouldResemble, propertiesResult)
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetUsers(ctx, params, nil)
			So(err, ShouldNotBeNil)
			So(returnedProperties, ShouldBeNil)
		})
	})

	Convey("when doing a batched request", t, func() {
		statMap := map[string]int{}
		userMap := map[string]*models.Properties{}
		identifiers := []string{}
		for i := 0; i < 250; i++ {
			identifier := strconv.Itoa(i)
			identifiers = append(identifiers, identifier)
			userMap[identifier] = &models.Properties{
				Login:       &identifier,
				ID:          identifier,
				Email:       &identifier,
				Displayname: &identifier,
				RemoteIP:    &identifier,
			}
		}
		ctx := context.Background()

		Convey("and there are no batches", func() {
			ts := initBatchTestServer(userMap, "id", statMap)
			defer ts.Close()
			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)

			users, err := c.GetUsers(ctx, &models.FilterParams{}, nil)

			So(err, ShouldBeNil)
			So(users.Results, ShouldBeEmpty)
			So(accessStatMap(statMap, requestsMade), ShouldEqual, 0)
		})

		Convey("and fetching multiple batches of IDs", func() {
			ts := initBatchTestServer(userMap, "id", statMap)
			defer ts.Close()
			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)

			users, err := c.GetUsers(ctx, &models.FilterParams{IDs: identifiers}, nil)

			So(err, ShouldBeNil)
			So(users, ShouldNotBeEmpty)
			So(len(users.Results), ShouldEqual, len(identifiers))
			So(accessStatMap(statMap, requestsMade), ShouldEqual, 3)
		})

		Convey("and fetching multiple batches where some fail", func() {
			ts := initBatchTestServer(userMap, "id", statMap)
			defer ts.Close()
			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)

			identifiers = append(identifiers, "")
			users, err := c.GetUsers(ctx, &models.FilterParams{IDs: identifiers}, nil)

			So(err, ShouldNotBeNil)
			So(users, ShouldBeNil)
			So(accessStatMap(statMap, requestErrors), ShouldEqual, 1)
		})

		Convey("and fetching multiple batches of logins", func() {
			ts := initBatchTestServer(userMap, "login", statMap)
			defer ts.Close()
			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)

			users, err := c.GetUsers(ctx, &models.FilterParams{Logins: identifiers}, nil)

			So(err, ShouldBeNil)
			So(users, ShouldNotBeEmpty)
			So(len(users.Results), ShouldEqual, len(identifiers))
			So(accessStatMap(statMap, requestsMade), ShouldEqual, 3)
		})

		Convey("and fetching multiple batches of emails", func() {
			ts := initBatchTestServer(userMap, "email", statMap)
			defer ts.Close()
			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)

			users, err := c.GetUsers(ctx, &models.FilterParams{Emails: identifiers}, nil)

			So(err, ShouldBeNil)
			So(users, ShouldNotBeEmpty)
			So(len(users.Results), ShouldEqual, len(identifiers))
			So(accessStatMap(statMap, requestsMade), ShouldEqual, 3)
		})

		Convey("and fetching multiple batches of display names", func() {
			ts := initBatchTestServer(userMap, "displayname", statMap)
			defer ts.Close()
			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)

			users, err := c.GetUsers(ctx, &models.FilterParams{DisplayNames: identifiers}, nil)

			So(err, ShouldBeNil)
			So(users, ShouldNotBeEmpty)
			So(len(users.Results), ShouldEqual, len(identifiers))
			So(accessStatMap(statMap, requestsMade), ShouldEqual, 3)
		})

		Convey("and fetching multiple batches of IPs", func() {
			ts := initBatchTestServer(userMap, "ip", statMap)
			defer ts.Close()
			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)

			users, err := c.GetUsers(ctx, &models.FilterParams{Ips: identifiers}, nil)

			So(err, ShouldBeNil)
			So(users, ShouldNotBeEmpty)
			So(len(users.Results), ShouldEqual, len(identifiers))
			So(accessStatMap(statMap, requestsMade), ShouldEqual, 3)
		})
	})
}

func TestGetBannedUsers(t *testing.T) {
	Convey("when calling GetUsers", t, func() {
		ctx := context.Background()

		untilTime, err := time.Parse(time.RFC3339, "2016-09-17T11:33:18-08:00")
		if err != nil {
			t.Errorf("Error parsing constant datetime string: %v", err)
		}

		ids := []string{"45573555"}
		logins := []string{"wimpy"}
		emails := []string{"services@twitch.tv"}
		query := "until=2016-09-17T11:33:18-08:00"
		propertiesResult := &models.PropertiesResult{
			Results: make([]*models.Properties, 0, 0),
		}
		for index, id := range ids {
			properties := &models.Properties{
				ID:    id,
				Login: &logins[index],
				Email: &emails[index],
			}
			propertiesResult.Results = append(propertiesResult.Results, properties)
		}

		Convey("when the backend is available", func() {
			ts := initBannedUsersTestServer(query, propertiesResult)
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetBannedUsers(ctx, untilTime, nil)
			So(err, ShouldBeNil)
			So(returnedProperties, ShouldResemble, propertiesResult)
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetBannedUsers(ctx, untilTime, nil)
			So(err, ShouldNotBeNil)
			So(returnedProperties, ShouldBeNil)
		})
	})
}

func TestBanUserByID(t *testing.T) {
	Convey("when calling BanUserByID", t, func() {
		ctx := context.Background()
		id := "45573555"
		reportingId := "129563558"
		banType := "tos"
		isWarning := true
		banReason := "FailFish"

		Convey("when the backend is available", func() {
			Convey("with a valid user ID", func() {
				ts := initBanTestServer(id)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				prop := &models.BanUserProperties{
					FromUserID:   reportingId,
					TargetUserID: id,
					Type:         banType,
					Reason:       banReason,
					Permanent:    !isWarning,
					SkipIPBan:    true,
				}

				err = c.BanUserByID(ctx, prop, nil)
				So(err, ShouldBeNil)
			})

			Convey("with a user ID that does not exist", func() {
				ts := initBanTestServer("69")
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				prop := &models.BanUserProperties{
					FromUserID:   reportingId,
					TargetUserID: id,
					Type:         banType,
					Reason:       banReason,
					Permanent:    !isWarning,
					SkipIPBan:    true,
				}

				err = c.BanUserByID(ctx, prop, nil)
				So(err, ShouldNotBeNil)
				So(err, ShouldHaveSameTypeAs, &UserNotFoundError{})
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			prop := &models.BanUserProperties{
				FromUserID:   reportingId,
				TargetUserID: id,
				Type:         banType,
				Reason:       banReason,
				Permanent:    !isWarning,
				SkipIPBan:    true,
			}
			err = c.BanUserByID(ctx, prop, nil)
			So(err, ShouldNotBeNil)
		})
	})
}

func TestSetUser(t *testing.T) {
	Convey("when calling SetUser", t, func() {
		ctx := context.Background()
		id := "45573555"
		email := "GarbagePete@garbage.com"
		uup := &models.UpdateableProperties{Email: &email}

		Convey("when the backend is available", func() {
			Convey("with a valid user ID", func() {
				ts := initSetUserTestServer(id)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				err = c.SetUser(ctx, id, uup, nil)
				So(err, ShouldBeNil)
			})

			Convey("with a user ID that does not exist", func() {
				ts := initSetUserTestServer("82")
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				err = c.SetUser(ctx, id, uup, nil)
				So(err, ShouldNotBeNil)
				So(err, ShouldHaveSameTypeAs, &UserNotFoundError{})
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			err = c.SetUser(ctx, id, uup, nil)
			So(err, ShouldNotBeNil)
		})
	})
}

func TestCreateUser(t *testing.T) {
	Convey("when calling CreateUser", t, func() {
		ctx := context.Background()
		newUserlogin := "newUserlogin"
		up := &models.CreateUserProperties{
			Login:    newUserlogin,
			Birthday: GetBirthday(AdminUserID),
			Email:    *Users[AdminUserID].Email,
		}

		Convey("when the backend is available", func() {
			Convey("with valid properties", func() {
				defaultCategory := "gaming"
				defaultLanguage := "en"
				newUser := models.Properties{
					ID:       "123",
					Login:    &newUserlogin,
					Birthday: Users[AdminUserID].Birthday,
					Email:    Users[AdminUserID].Email,
					Language: &defaultLanguage,
					Category: &defaultCategory,
				}
				ts := initCreateUserTestServer(&newUser)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				prop, err := c.CreateUser(ctx, up, nil)
				So(err, ShouldBeNil)
				So(prop, ShouldNotBeNil)
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			_, err = c.CreateUser(ctx, up, nil)
			So(err, ShouldNotBeNil)
		})
	})
}

func TestUnbanUserByID(t *testing.T) {
	Convey("when calling UnbanUserByID", t, func() {
		ctx := context.Background()
		id := "45573555"
		Convey("when the backend is available", func() {
			Convey("with a valid user ID", func() {
				ts := initBanTestServer(id)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				err = c.UnbanUserByID(ctx, id, false, nil)
				So(err, ShouldBeNil)
			})

			Convey("with a user ID that does not exist", func() {
				ts := initBanTestServer("69")
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				err = c.UnbanUserByID(ctx, id, false, nil)
				So(err, ShouldNotBeNil)
				So(err, ShouldHaveSameTypeAs, &UserNotFoundError{})
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			err = c.UnbanUserByID(ctx, id, false, nil)
			So(err, ShouldNotBeNil)
		})
	})
}

func TestAddDMCAStrike(t *testing.T) {
	Convey("when calling AddDMCAStrike", t, func() {
		ctx := context.Background()
		id := "45573555"
		Convey("when the backend is available", func() {
			Convey("with a valid user ID", func() {
				ts := initDMCAStrikeTestServer(id)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				err = c.AddDMCAStrike(ctx, id, nil)
				So(err, ShouldBeNil)
			})

			Convey("with a user ID that does not exist", func() {
				ts := initDMCAStrikeTestServer("69")
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				err = c.AddDMCAStrike(ctx, id, nil)
				So(err, ShouldNotBeNil)
				So(err, ShouldHaveSameTypeAs, &UserNotFoundError{})
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			err = c.AddDMCAStrike(ctx, id, nil)
			So(err, ShouldNotBeNil)
		})
	})
}
func TestRemoveDMCAStrike(t *testing.T) {
	Convey("when calling RemoveDMCAStrike", t, func() {
		ctx := context.Background()
		id := "45573555"
		Convey("when the backend is available", func() {
			Convey("with a valid user ID", func() {
				ts := initDMCAStrikeTestServer(id)
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				err = c.RemoveDMCAStrike(ctx, id, nil)
				So(err, ShouldBeNil)
			})

			Convey("with a user ID that does not exist", func() {
				ts := initDMCAStrikeTestServer("69")
				defer ts.Close()

				c, err := NewClient(twitchclient.ClientConf{
					Host: ts.URL,
				})
				So(err, ShouldBeNil)
				So(c, ShouldNotBeNil)

				err = c.RemoveDMCAStrike(ctx, id, nil)
				So(err, ShouldNotBeNil)
				So(err, ShouldHaveSameTypeAs, &UserNotFoundError{})
			})
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			err = c.RemoveDMCAStrike(ctx, id, nil)
			So(err, ShouldNotBeNil)
		})
	})
}

func TestGetGlobalPrivilegedUsers(t *testing.T) {
	Convey("when calling GetGlobalPrivilegedUsers", t, func() {
		ctx := context.Background()

		roles := []string{"admin", "global_mod"}

		query := "role=admin&role=global_mod"
		privilegedUsersResult := &models.GlobalPrivilegedUsers{
			Admins:     []string{"pipe", "extalix", "lucent_moon", "killjoysnr"},
			GlobalMods: []string{"nox", "kazzy"},
		}

		Convey("when the backend is available", func() {
			ts := initPrivilegedUsersTestServer(query, privilegedUsersResult)
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetGlobalPrivilegedUsers(ctx, roles, nil)
			So(err, ShouldBeNil)
			So(returnedProperties, ShouldResemble, privilegedUsersResult)
		})

		Convey("when the backend is dead", func() {
			ts := initBrokenTestServer()
			defer ts.Close()

			c, err := NewClient(twitchclient.ClientConf{
				Host: ts.URL,
			})
			So(err, ShouldBeNil)
			So(c, ShouldNotBeNil)

			returnedProperties, err := c.GetGlobalPrivilegedUsers(ctx, roles, nil)
			So(err, ShouldNotBeNil)
			So(returnedProperties, ShouldBeNil)
		})
	})
}

func initBrokenTestServer() *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))
}

func initDMCAStrikeTestServer(id string) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.URL.String() != fmt.Sprintf("/users/%s/dmca_strike", id) {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		w.WriteHeader(http.StatusNoContent)
	}))
}

func initBanTestServer(id string) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.URL.String() != fmt.Sprintf("/users/%s/ban", id) {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		w.WriteHeader(http.StatusNoContent)
	}))
}

func initSetUserTestServer(id string) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.URL.String() != fmt.Sprintf("/users/%s", id) {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		w.WriteHeader(http.StatusNoContent)
	}))
}

func initCreateUserTestServer(user *models.Properties) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.URL.String() != "/users" {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		err := json.NewEncoder(w).Encode(user)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		} else {
			w.WriteHeader(http.StatusOK)
		}
	}))
}

func initUserTestServer(id string, user *models.Properties) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.URL.String() != fmt.Sprintf("/users/%s?return_id_as_string=true", id) {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		err := json.NewEncoder(w).Encode(user)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}))
}

func initUsersTestServer(query string, users *models.PropertiesResult) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.URL.String() != fmt.Sprintf("/users?%s&return_id_as_string=true", query) {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		err := json.NewEncoder(w).Encode(users)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}))
}

func initBannedUsersTestServer(query string, users *models.PropertiesResult) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		unescaped, err := url.QueryUnescape(r.URL.String())
		if err != nil {
			panic("Fix the query string please\n")
		}
		if unescaped != fmt.Sprintf("/banned_users?return_id_as_string=true&%s", query) {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		err = json.NewEncoder(w).Encode(users)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}))
}

func initPrivilegedUsersTestServer(query string, users *models.GlobalPrivilegedUsers) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.URL.String() != fmt.Sprintf("/global_privileged_users?%s", query) {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		err := json.NewEncoder(w).Encode(users)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}))
}

var mutex sync.Mutex

// necessary to prevent testing race condition failure
func accessStatMap(statMap map[string]int, key string) int {
	mutex.Lock()
	defer mutex.Unlock()

	return statMap[key]
}

func initBatchTestServer(userMap map[string]*models.Properties, key string, statMap map[string]int) *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		mutex.Lock()
		defer mutex.Unlock()

		statMap[requestsMade]++
		users := []*models.Properties{}
		identifiers := r.URL.Query()[key]
		if len(identifiers) > BatchSize {
			panic(fmt.Errorf("batching did not occur: expected less than or equal to %d elements, received %d", BatchSize, len(identifiers)))
		}
		for _, identifier := range identifiers {
			if identifier == "" {
				statMap[requestErrors]++
				w.WriteHeader(http.StatusInternalServerError)
				return
			}
			user, ok := userMap[identifier]
			if ok {
				users = append(users, user)
			}
		}

		result := &models.PropertiesResult{Results: users}

		err := json.NewEncoder(w).Encode(result)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}))
}

func TestIsServerError(t *testing.T) {
	for name, scenario := range map[string]struct {
		Error  error
		Result bool
	}{
		"user_not_found_error": {
			Error: &UserNotFoundError{},
		},
		"model_error": {
			Error: models.ErrFilteredUser,
		},
		"error_response": {
			Error: models.ErrorResponse{
				Status: http.StatusInternalServerError,
			},
			Result: true,
		},
	} {
		t.Run(name, func(t *testing.T) {
			actual := IsInternalServerError(scenario.Error)
			if actual != scenario.Result {
				t.Errorf("expected %t, got %t", scenario.Result, actual)
			}
		})
	}
}
