package main

import (
	"bufio"
	"encoding/csv"
	"flag"
	"fmt"
	"io"
	"os"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/foundation/history-service/client/history"
	auditormodels "code.justin.tv/web/users-service/cmd/auditor-refill"
	"code.justin.tv/web/users-service/internal/clients/auditor"
	"code.justin.tv/web/users-service/internal/clients/auditor/events"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	AUDITOR_PROD_HOST    = "http://history.prod.us-west2.twitch.tv/"
	AUDITOR_STAGING_HOST = "http://history.staging.us-west2.twitch.tv/"
)

func main() {
	ctx := context.Background()
	var file string
	var ifTest bool
	flag.StringVar(&file, "file", "", "file path")
	flag.BoolVar(&ifTest, "if testing", true, "if refill data to staging for testing")
	flag.Parse()

	host := AUDITOR_STAGING_HOST
	if !ifTest {
		host = AUDITOR_PROD_HOST
	}

	auditorClient, err := configureAuditor(host)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to configure auditor:", err))
		return
	}

	outputf, err := os.Create("result")
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to create output file:", err))
	}
	w := bufio.NewWriter(outputf)

	defer func() {
		err = w.Flush()
		if err != nil {
			logx.Fatal(ctx, fmt.Sprint("failed to flush writer:", err))
		}
		err = outputf.Close()
		if err != nil {
			logx.Fatal(ctx, fmt.Sprint("failed to close output file:", err))
		}
	}()

	form := "January 2, 2006"
	// Currently start time and endtime is not configurable in command. We'd like to see if we have further needs on this script, if yes, we could make it in input fields and make the script more generalized for reuse.
	startTime, err := time.Parse(form, "May 31, 2017")
	if err != nil {
		fmt.Fprint(w, "Fail to parse start time.")
	}

	endTime, err := time.Parse(form, "July 10, 2017")
	if err != nil {
		fmt.Fprint(w, "Fail to parse end time.")
	}

	f, err := os.Open(file)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to open source file:", err))
	}
	r := csv.NewReader(bufio.NewReader(f))
	for {
		record, err := r.Read()
		if err == io.EOF {
			break
		}

		if len(record) != 4 {
			fmt.Fprint(w, "bad record")
			break
		}

		input := auditormodels.RenameInput{
			UserID:   record[1],
			OldValue: record[2],
			NewValue: record[3],
		}

		// Iteration to see if the previous specific records has the same new value
		// Call auditor
		result, err := auditorClient.Search(ctx, &history.SearchParams{
			Action:       auditor.AttributeUpdate,
			UserType:     auditor.DefaultUserType,
			UserID:       input.UserID,
			ResourceType: auditor.DefaultUserType,
			ResourceID:   input.UserID,
			CreatedAtLte: endTime,
			CreatedAtGte: startTime,
			Attribute:    "login",
			OldValue:     input.OldValue,
			NewValue:     input.NewValue,
		})

		if len(result.Audits) == 0 || err != nil {
			// If not exist in current history, refill it and POST to auditor
			updateEvents := events.UpdateUserEvent(&models.Properties{
				ID:    input.UserID,
				Login: &input.OldValue,
			}, &models.UpdateableProperties{
				ID:       input.UserID,
				NewLogin: &input.NewValue,
			})
			createTime, err := time.Parse("1/2/06  15:04", record[0])
			if err != nil {
				fmt.Fprintf(w, "time parsing error %v for user id %d\n", err, input.UserID)
				continue
			}
			updateEvents.CreatedAt = &createTime
			auditorClient.Audit(ctx, updateEvents)
			fmt.Fprintf(w, "rerill for user id %s, %s -> %s on %s\n", input.UserID, input.OldValue, input.NewValue, record[0])
		}
	}
	err = w.Flush()
	if err != nil {
		logx.Fatal(ctx, fmt.Sprint("failed to flush output file:", err))
	}
}

func configureAuditor(host string) (auditor.Auditor, error) {
	conf := twitchhttp.ClientConf{
		Host: host,
	}
	return auditor.NewAuditor(conf)
}
