package configs

import (
	"encoding/json"
	"fmt"
	"strings"
	"time"
)

// MultiCache is a generated multi region wrapper for Cache.
type MultiCache struct {
	Primary MultiRedis
	Backup  MultiRedis
}

// Value determines the values for each field given an environment and region
func (m MultiCache) Value(env, region string) Cache {
	return Cache{
		Primary: m.Primary.Value(env, region),
		Backup:  m.Backup.Value(env, region),
	}
}

// MultiConfig is a generated multi region wrapper for Config.
type MultiConfig struct {
	SiteDB               MultiDB
	Reservations         MultiDB
	UserMutations        MultiKinesisStream
	Topics               MultiTopics
	Hosts                MultiHosts
	LegacyAWSKey         EnvString
	LegacyAWSSecret      EnvString
	XraySampling         EnvFloat64
	Twilio               MultiTwilio
	GeoIPPath            EnvString
	CartmanKeyPath       EnvString
	SQSRegion            EnvString
	SQSEndpoint          EnvString
	NoopRailsClient      EnvBool
	STSAssumeRole        EnvString
	Workers              MultiWorkerQueues
	RollbarToken         EnvString
	Leviathan            MultiLeviathan
	Cache                MultiCache
	RedisTTL             EnvInt
	SeedVerifyCode       EnvString
	Stats                MultiStats
	HystrixSampling      EnvFloat32
	HystrixErrorSampling EnvFloat32
}

// Value determines the values for each field given an environment and region
func (m MultiConfig) Value(env, region string) Config {
	return Config{
		SiteDB:               m.SiteDB.Value(env, region),
		Reservations:         m.Reservations.Value(env, region),
		UserMutations:        m.UserMutations.Value(env, region),
		Topics:               m.Topics.Value(env, region),
		Hosts:                m.Hosts.Value(env, region),
		LegacyAWSKey:         m.LegacyAWSKey.Value(env, region),
		LegacyAWSSecret:      m.LegacyAWSSecret.Value(env, region),
		XraySampling:         m.XraySampling.Value(env, region),
		Twilio:               m.Twilio.Value(env, region),
		GeoIPPath:            m.GeoIPPath.Value(env, region),
		CartmanKeyPath:       m.CartmanKeyPath.Value(env, region),
		SQSRegion:            m.SQSRegion.Value(env, region),
		SQSEndpoint:          m.SQSEndpoint.Value(env, region),
		NoopRailsClient:      m.NoopRailsClient.Value(env, region),
		STSAssumeRole:        m.STSAssumeRole.Value(env, region),
		Workers:              m.Workers.Value(env, region),
		RollbarToken:         m.RollbarToken.Value(env, region),
		Leviathan:            m.Leviathan.Value(env, region),
		Cache:                m.Cache.Value(env, region),
		RedisTTL:             m.RedisTTL.Value(env, region),
		SeedVerifyCode:       m.SeedVerifyCode.Value(env, region),
		Stats:                m.Stats.Value(env, region),
		HystrixSampling:      m.HystrixSampling.Value(env, region),
		HystrixErrorSampling: m.HystrixErrorSampling.Value(env, region),
	}
}

// MultiDB is a generated multi region wrapper for DB.
type MultiDB struct {
	Name         EnvString
	User         EnvString
	Password     EnvString
	PasswordFile EnvString
	Slave        MultiDBHost
	Master       MultiDBHost
}

// Value determines the values for each field given an environment and region
func (m MultiDB) Value(env, region string) DB {
	return DB{
		Name:         m.Name.Value(env, region),
		User:         m.User.Value(env, region),
		Password:     m.Password.Value(env, region),
		PasswordFile: m.PasswordFile.Value(env, region),
		Slave:        m.Slave.Value(env, region),
		Master:       m.Master.Value(env, region),
	}
}

// MultiDBHost is a generated multi region wrapper for DBHost.
type MultiDBHost struct {
	Host               EnvString
	Port               EnvInt
	MaxOpenConns       EnvInt
	MaxIdleConns       EnvInt
	MaxQueueSize       EnvInt
	ConnAcquireTimeout EnvDuration
	RequestTimeout     EnvDuration
	MaxConnAge         EnvDuration
	LoggerPrefix       EnvString
}

// Value determines the values for each field given an environment and region
func (m MultiDBHost) Value(env, region string) DBHost {
	return DBHost{
		Host:               m.Host.Value(env, region),
		Port:               m.Port.Value(env, region),
		MaxOpenConns:       m.MaxOpenConns.Value(env, region),
		MaxIdleConns:       m.MaxIdleConns.Value(env, region),
		MaxQueueSize:       m.MaxQueueSize.Value(env, region),
		ConnAcquireTimeout: m.ConnAcquireTimeout.Value(env, region),
		RequestTimeout:     m.RequestTimeout.Value(env, region),
		MaxConnAge:         m.MaxConnAge.Value(env, region),
		LoggerPrefix:       m.LoggerPrefix.Value(env, region),
	}
}

// MultiHosts is a generated multi region wrapper for Hosts.
type MultiHosts struct {
	Follows       EnvString
	Partnerships  EnvString
	Spade         EnvString
	Auditor       EnvString
	Discovery     EnvString
	EVS           EnvString
	UploadService EnvString
	Owl           EnvString
	PubSub        EnvString
	Rails         EnvString
}

// Value determines the values for each field given an environment and region
func (m MultiHosts) Value(env, region string) Hosts {
	return Hosts{
		Follows:       m.Follows.Value(env, region),
		Partnerships:  m.Partnerships.Value(env, region),
		Spade:         m.Spade.Value(env, region),
		Auditor:       m.Auditor.Value(env, region),
		Discovery:     m.Discovery.Value(env, region),
		EVS:           m.EVS.Value(env, region),
		UploadService: m.UploadService.Value(env, region),
		Owl:           m.Owl.Value(env, region),
		PubSub:        m.PubSub.Value(env, region),
		Rails:         m.Rails.Value(env, region),
	}
}

// MultiKinesisStream is a generated multi region wrapper for KinesisStream.
type MultiKinesisStream struct {
	Role       EnvString
	Name       EnvString
	Region     EnvString
	RetryCount EnvInt
	RetryDelay EnvInt
}

// Value determines the values for each field given an environment and region
func (m MultiKinesisStream) Value(env, region string) KinesisStream {
	return KinesisStream{
		Role:       m.Role.Value(env, region),
		Name:       m.Name.Value(env, region),
		Region:     m.Region.Value(env, region),
		RetryCount: m.RetryCount.Value(env, region),
		RetryDelay: m.RetryDelay.Value(env, region),
	}
}

// MultiLeviathan is a generated multi region wrapper for Leviathan.
type MultiLeviathan struct {
	Host  EnvString
	Token EnvString
}

// Value determines the values for each field given an environment and region
func (m MultiLeviathan) Value(env, region string) Leviathan {
	return Leviathan{
		Host:  m.Host.Value(env, region),
		Token: m.Token.Value(env, region),
	}
}

// MultiRedis is a generated multi region wrapper for Redis.
type MultiRedis struct {
	Host           EnvString
	Password       EnvString
	UseClusterMode EnvBool
	ConnectTimeout EnvInt
	ReadTimeout    EnvInt
	WriteTimeout   EnvInt
	MaxConnections EnvInt
}

// Value determines the values for each field given an environment and region
func (m MultiRedis) Value(env, region string) Redis {
	return Redis{
		Host:           m.Host.Value(env, region),
		Password:       m.Password.Value(env, region),
		UseClusterMode: m.UseClusterMode.Value(env, region),
		ConnectTimeout: m.ConnectTimeout.Value(env, region),
		ReadTimeout:    m.ReadTimeout.Value(env, region),
		WriteTimeout:   m.WriteTimeout.Value(env, region),
		MaxConnections: m.MaxConnections.Value(env, region),
	}
}

// MultiStats is a generated multi region wrapper for Stats.
type MultiStats struct {
	HostPort EnvString
	App      EnvString
}

// Value determines the values for each field given an environment and region
func (m MultiStats) Value(env, region string) Stats {
	return Stats{
		HostPort: m.HostPort.Value(env, region),
		App:      m.App.Value(env, region),
	}
}

// MultiTopics is a generated multi region wrapper for Topics.
type MultiTopics struct {
	SNSRegions            EnvString
	ModerationEvents      EnvString
	RenameEvents          EnvString
	CreationEvents        EnvString
	MutationEvents        EnvString
	ImageUploadEvents     EnvString
	ChannelMutationEvents EnvString
	EmailVerified         EnvString
	PushyDispatched       EnvString
	UserSoftDeleteEvents  EnvString
	UserUndeleteEvents    EnvString
	UserHardDeleteEvents  EnvString
	ExpireCacheEvents     EnvString
}

// Value determines the values for each field given an environment and region
func (m MultiTopics) Value(env, region string) Topics {
	return Topics{
		SNSRegions:            m.SNSRegions.Value(env, region),
		ModerationEvents:      m.ModerationEvents.Value(env, region),
		RenameEvents:          m.RenameEvents.Value(env, region),
		CreationEvents:        m.CreationEvents.Value(env, region),
		MutationEvents:        m.MutationEvents.Value(env, region),
		ImageUploadEvents:     m.ImageUploadEvents.Value(env, region),
		ChannelMutationEvents: m.ChannelMutationEvents.Value(env, region),
		EmailVerified:         m.EmailVerified.Value(env, region),
		PushyDispatched:       m.PushyDispatched.Value(env, region),
		UserSoftDeleteEvents:  m.UserSoftDeleteEvents.Value(env, region),
		UserUndeleteEvents:    m.UserUndeleteEvents.Value(env, region),
		UserHardDeleteEvents:  m.UserHardDeleteEvents.Value(env, region),
		ExpireCacheEvents:     m.ExpireCacheEvents.Value(env, region),
	}
}

// MultiTwilio is a generated multi region wrapper for Twilio.
type MultiTwilio struct {
	Account      EnvString
	Auth         EnvString
	PhoneNumbers EnvString
	Disabled     EnvBool
}

// Value determines the values for each field given an environment and region
func (m MultiTwilio) Value(env, region string) Twilio {
	return Twilio{
		Account:      m.Account.Value(env, region),
		Auth:         m.Auth.Value(env, region),
		PhoneNumbers: m.PhoneNumbers.Value(env, region),
		Disabled:     m.Disabled.Value(env, region),
	}
}

// MultiWorkerQueue is a generated multi region wrapper for WorkerQueue.
type MultiWorkerQueue struct {
	NumWorkers           EnvInt
	QueueName            EnvString
	AccountNumber        EnvString
	QueueEndpoint        EnvString
	LogEventName         EnvString
	Disabled             EnvBool
	MaxVisibilityTimeout EnvInt
}

// Value determines the values for each field given an environment and region
func (m MultiWorkerQueue) Value(env, region string) WorkerQueue {
	return WorkerQueue{
		NumWorkers:           m.NumWorkers.Value(env, region),
		QueueName:            m.QueueName.Value(env, region),
		AccountNumber:        m.AccountNumber.Value(env, region),
		QueueEndpoint:        m.QueueEndpoint.Value(env, region),
		LogEventName:         m.LogEventName.Value(env, region),
		Disabled:             m.Disabled.Value(env, region),
		MaxVisibilityTimeout: m.MaxVisibilityTimeout.Value(env, region),
	}
}

// MultiWorkerQueues is a generated multi region wrapper for WorkerQueues.
type MultiWorkerQueues struct {
	EmailVerified MultiWorkerQueue
	ImageUpload   MultiWorkerQueue
	DeadLetter    MultiWorkerQueue
	ExpireCache   MultiWorkerQueue
}

// Value determines the values for each field given an environment and region
func (m MultiWorkerQueues) Value(env, region string) WorkerQueues {
	return WorkerQueues{
		EmailVerified: m.EmailVerified.Value(env, region),
		ImageUpload:   m.ImageUpload.Value(env, region),
		DeadLetter:    m.DeadLetter.Value(env, region),
		ExpireCache:   m.ExpireCache.Value(env, region),
	}
}

// EnvBool is a multi environment, region wrapper for bool
type EnvBool struct {
	Default     bool
	Production  *RegionBool
	Staging     *RegionBool
	Development *RegionBool
	Test        *RegionBool
}

// Value determines the value given an environment and region.
func (e EnvBool) Value(env, region string) bool {
	var found *RegionBool
	var value *bool

	switch strings.ToLower(env) {
	case "production":
		found = e.Production
	case "staging":
		found = e.Staging
	case "development":
		found = e.Development
	case "test":
		found = e.Test
	}

	if found != nil {
		value = found.Value(region)
	}

	if value != nil {
		return *value
	}

	return e.Default
}

// UnmarshalJSON unmarshals either a bool into Default or map of RegionBools
func (e *EnvBool) UnmarshalJSON(b []byte) error {
	var def bool
	if err := json.Unmarshal(b, &def); err == nil {
		e.Default = def
		return nil
	}

	m := make(map[string]*RegionBool)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		if value == nil {
			continue
		}
		switch strings.ToLower(key) {
		case "default":
			e.Default = *value.Default
		case "production":
			e.Production = value
		case "staging":
			e.Staging = value
		case "development":
			e.Development = value
		case "test":
			e.Test = value
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'production', 'staging', 'development', 'test'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (e EnvBool) MarshalJSON() ([]byte, error) {
	if e.Production == nil && e.Staging == nil && e.Development == nil && e.Test == nil {
		return json.Marshal(e.Default)
	}

	m := make(map[string]interface{})
	m["default"] = e.Default
	if e.Production != nil {
		m["production"] = e.Production
	}
	if e.Staging != nil {
		m["staging"] = e.Staging
	}
	if e.Development != nil {
		m["development"] = e.Development
	}
	if e.Test != nil {
		m["test"] = e.Test
	}

	return json.Marshal(m)
}

// RegionBool is a multi region wrapper for bool.
type RegionBool struct {
	Default *bool
	USWest2 *bool
	USEast1 *bool
}

// Value determines the value given a region.
func (r RegionBool) Value(region string) *bool {
	var found *bool

	switch strings.ToLower(region) {
	case "us-west-2":
		found = r.USWest2
	case "us-east-1":
		found = r.USEast1
	}

	if found != nil {
		return found
	}

	return r.Default
}

// UnmarshalJSON unmarshals either a bool into Default or map of bools
func (r *RegionBool) UnmarshalJSON(b []byte) error {
	var def bool
	if err := json.Unmarshal(b, &def); err == nil {
		r.Default = &def
		return nil
	}

	m := make(map[string]bool)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		loopSafePtr := value
		switch strings.ToLower(key) {
		case "default":
			r.Default = &loopSafePtr
		case "us-west-2":
			r.USWest2 = &loopSafePtr
		case "us-east-1":
			r.USEast1 = &loopSafePtr
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'us-west-2', 'us-east-1'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (r RegionBool) MarshalJSON() ([]byte, error) {
	if r.USWest2 == nil && r.USEast1 == nil {
		return json.Marshal(r.Default)
	}

	m := make(map[string]bool)
	if r.Default != nil {
		m["default"] = *r.Default
	}
	if r.USWest2 != nil {
		m["us-west-2"] = *r.USWest2
	}
	if r.USEast1 != nil {
		m["us-east-1"] = *r.USEast1
	}

	return json.Marshal(m)
}

// EnvFloat32 is a multi environment, region wrapper for float32
type EnvFloat32 struct {
	Default     float32
	Production  *RegionFloat32
	Staging     *RegionFloat32
	Development *RegionFloat32
	Test        *RegionFloat32
}

// Value determines the value given an environment and region.
func (e EnvFloat32) Value(env, region string) float32 {
	var found *RegionFloat32
	var value *float32

	switch strings.ToLower(env) {
	case "production":
		found = e.Production
	case "staging":
		found = e.Staging
	case "development":
		found = e.Development
	case "test":
		found = e.Test
	}

	if found != nil {
		value = found.Value(region)
	}

	if value != nil {
		return *value
	}

	return e.Default
}

// UnmarshalJSON unmarshals either a float32 into Default or map of RegionFloat32s
func (e *EnvFloat32) UnmarshalJSON(b []byte) error {
	var def float32
	if err := json.Unmarshal(b, &def); err == nil {
		e.Default = def
		return nil
	}

	m := make(map[string]*RegionFloat32)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		if value == nil {
			continue
		}
		switch strings.ToLower(key) {
		case "default":
			e.Default = *value.Default
		case "production":
			e.Production = value
		case "staging":
			e.Staging = value
		case "development":
			e.Development = value
		case "test":
			e.Test = value
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'production', 'staging', 'development', 'test'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (e EnvFloat32) MarshalJSON() ([]byte, error) {
	if e.Production == nil && e.Staging == nil && e.Development == nil && e.Test == nil {
		return json.Marshal(e.Default)
	}

	m := make(map[string]interface{})
	m["default"] = e.Default
	if e.Production != nil {
		m["production"] = e.Production
	}
	if e.Staging != nil {
		m["staging"] = e.Staging
	}
	if e.Development != nil {
		m["development"] = e.Development
	}
	if e.Test != nil {
		m["test"] = e.Test
	}

	return json.Marshal(m)
}

// RegionFloat32 is a multi region wrapper for float32.
type RegionFloat32 struct {
	Default *float32
	USWest2 *float32
	USEast1 *float32
}

// Value determines the value given a region.
func (r RegionFloat32) Value(region string) *float32 {
	var found *float32

	switch strings.ToLower(region) {
	case "us-west-2":
		found = r.USWest2
	case "us-east-1":
		found = r.USEast1
	}

	if found != nil {
		return found
	}

	return r.Default
}

// UnmarshalJSON unmarshals either a float32 into Default or map of float32s
func (r *RegionFloat32) UnmarshalJSON(b []byte) error {
	var def float32
	if err := json.Unmarshal(b, &def); err == nil {
		r.Default = &def
		return nil
	}

	m := make(map[string]float32)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		loopSafePtr := value
		switch strings.ToLower(key) {
		case "default":
			r.Default = &loopSafePtr
		case "us-west-2":
			r.USWest2 = &loopSafePtr
		case "us-east-1":
			r.USEast1 = &loopSafePtr
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'us-west-2', 'us-east-1'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (r RegionFloat32) MarshalJSON() ([]byte, error) {
	if r.USWest2 == nil && r.USEast1 == nil {
		return json.Marshal(r.Default)
	}

	m := make(map[string]float32)
	if r.Default != nil {
		m["default"] = *r.Default
	}
	if r.USWest2 != nil {
		m["us-west-2"] = *r.USWest2
	}
	if r.USEast1 != nil {
		m["us-east-1"] = *r.USEast1
	}

	return json.Marshal(m)
}

// EnvFloat64 is a multi environment, region wrapper for float64
type EnvFloat64 struct {
	Default     float64
	Production  *RegionFloat64
	Staging     *RegionFloat64
	Development *RegionFloat64
	Test        *RegionFloat64
}

// Value determines the value given an environment and region.
func (e EnvFloat64) Value(env, region string) float64 {
	var found *RegionFloat64
	var value *float64

	switch strings.ToLower(env) {
	case "production":
		found = e.Production
	case "staging":
		found = e.Staging
	case "development":
		found = e.Development
	case "test":
		found = e.Test
	}

	if found != nil {
		value = found.Value(region)
	}

	if value != nil {
		return *value
	}

	return e.Default
}

// UnmarshalJSON unmarshals either a float64 into Default or map of RegionFloat64s
func (e *EnvFloat64) UnmarshalJSON(b []byte) error {
	var def float64
	if err := json.Unmarshal(b, &def); err == nil {
		e.Default = def
		return nil
	}

	m := make(map[string]*RegionFloat64)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		if value == nil {
			continue
		}
		switch strings.ToLower(key) {
		case "default":
			e.Default = *value.Default
		case "production":
			e.Production = value
		case "staging":
			e.Staging = value
		case "development":
			e.Development = value
		case "test":
			e.Test = value
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'production', 'staging', 'development', 'test'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (e EnvFloat64) MarshalJSON() ([]byte, error) {
	if e.Production == nil && e.Staging == nil && e.Development == nil && e.Test == nil {
		return json.Marshal(e.Default)
	}

	m := make(map[string]interface{})
	m["default"] = e.Default
	if e.Production != nil {
		m["production"] = e.Production
	}
	if e.Staging != nil {
		m["staging"] = e.Staging
	}
	if e.Development != nil {
		m["development"] = e.Development
	}
	if e.Test != nil {
		m["test"] = e.Test
	}

	return json.Marshal(m)
}

// RegionFloat64 is a multi region wrapper for float64.
type RegionFloat64 struct {
	Default *float64
	USWest2 *float64
	USEast1 *float64
}

// Value determines the value given a region.
func (r RegionFloat64) Value(region string) *float64 {
	var found *float64

	switch strings.ToLower(region) {
	case "us-west-2":
		found = r.USWest2
	case "us-east-1":
		found = r.USEast1
	}

	if found != nil {
		return found
	}

	return r.Default
}

// UnmarshalJSON unmarshals either a float64 into Default or map of float64s
func (r *RegionFloat64) UnmarshalJSON(b []byte) error {
	var def float64
	if err := json.Unmarshal(b, &def); err == nil {
		r.Default = &def
		return nil
	}

	m := make(map[string]float64)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		loopSafePtr := value
		switch strings.ToLower(key) {
		case "default":
			r.Default = &loopSafePtr
		case "us-west-2":
			r.USWest2 = &loopSafePtr
		case "us-east-1":
			r.USEast1 = &loopSafePtr
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'us-west-2', 'us-east-1'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (r RegionFloat64) MarshalJSON() ([]byte, error) {
	if r.USWest2 == nil && r.USEast1 == nil {
		return json.Marshal(r.Default)
	}

	m := make(map[string]float64)
	if r.Default != nil {
		m["default"] = *r.Default
	}
	if r.USWest2 != nil {
		m["us-west-2"] = *r.USWest2
	}
	if r.USEast1 != nil {
		m["us-east-1"] = *r.USEast1
	}

	return json.Marshal(m)
}

// EnvInt is a multi environment, region wrapper for int
type EnvInt struct {
	Default     int
	Production  *RegionInt
	Staging     *RegionInt
	Development *RegionInt
	Test        *RegionInt
}

// Value determines the value given an environment and region.
func (e EnvInt) Value(env, region string) int {
	var found *RegionInt
	var value *int

	switch strings.ToLower(env) {
	case "production":
		found = e.Production
	case "staging":
		found = e.Staging
	case "development":
		found = e.Development
	case "test":
		found = e.Test
	}

	if found != nil {
		value = found.Value(region)
	}

	if value != nil {
		return *value
	}

	return e.Default
}

// UnmarshalJSON unmarshals either a int into Default or map of RegionInts
func (e *EnvInt) UnmarshalJSON(b []byte) error {
	var def int
	if err := json.Unmarshal(b, &def); err == nil {
		e.Default = def
		return nil
	}

	m := make(map[string]*RegionInt)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		if value == nil {
			continue
		}
		switch strings.ToLower(key) {
		case "default":
			e.Default = *value.Default
		case "production":
			e.Production = value
		case "staging":
			e.Staging = value
		case "development":
			e.Development = value
		case "test":
			e.Test = value
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'production', 'staging', 'development', 'test'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (e EnvInt) MarshalJSON() ([]byte, error) {
	if e.Production == nil && e.Staging == nil && e.Development == nil && e.Test == nil {
		return json.Marshal(e.Default)
	}

	m := make(map[string]interface{})
	m["default"] = e.Default
	if e.Production != nil {
		m["production"] = e.Production
	}
	if e.Staging != nil {
		m["staging"] = e.Staging
	}
	if e.Development != nil {
		m["development"] = e.Development
	}
	if e.Test != nil {
		m["test"] = e.Test
	}

	return json.Marshal(m)
}

// RegionInt is a multi region wrapper for int.
type RegionInt struct {
	Default *int
	USWest2 *int
	USEast1 *int
}

// Value determines the value given a region.
func (r RegionInt) Value(region string) *int {
	var found *int

	switch strings.ToLower(region) {
	case "us-west-2":
		found = r.USWest2
	case "us-east-1":
		found = r.USEast1
	}

	if found != nil {
		return found
	}

	return r.Default
}

// UnmarshalJSON unmarshals either a int into Default or map of ints
func (r *RegionInt) UnmarshalJSON(b []byte) error {
	var def int
	if err := json.Unmarshal(b, &def); err == nil {
		r.Default = &def
		return nil
	}

	m := make(map[string]int)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		loopSafePtr := value
		switch strings.ToLower(key) {
		case "default":
			r.Default = &loopSafePtr
		case "us-west-2":
			r.USWest2 = &loopSafePtr
		case "us-east-1":
			r.USEast1 = &loopSafePtr
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'us-west-2', 'us-east-1'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (r RegionInt) MarshalJSON() ([]byte, error) {
	if r.USWest2 == nil && r.USEast1 == nil {
		return json.Marshal(r.Default)
	}

	m := make(map[string]int)
	if r.Default != nil {
		m["default"] = *r.Default
	}
	if r.USWest2 != nil {
		m["us-west-2"] = *r.USWest2
	}
	if r.USEast1 != nil {
		m["us-east-1"] = *r.USEast1
	}

	return json.Marshal(m)
}

// EnvString is a multi environment, region wrapper for string
type EnvString struct {
	Default     string
	Production  *RegionString
	Staging     *RegionString
	Development *RegionString
	Test        *RegionString
}

// Value determines the value given an environment and region.
func (e EnvString) Value(env, region string) string {
	var found *RegionString
	var value *string

	switch strings.ToLower(env) {
	case "production":
		found = e.Production
	case "staging":
		found = e.Staging
	case "development":
		found = e.Development
	case "test":
		found = e.Test
	}

	if found != nil {
		value = found.Value(region)
	}

	if value != nil {
		return *value
	}

	return e.Default
}

// UnmarshalJSON unmarshals either a string into Default or map of RegionStrings
func (e *EnvString) UnmarshalJSON(b []byte) error {
	var def string
	if err := json.Unmarshal(b, &def); err == nil {
		e.Default = def
		return nil
	}

	m := make(map[string]*RegionString)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		if value == nil {
			continue
		}
		switch strings.ToLower(key) {
		case "default":
			e.Default = *value.Default
		case "production":
			e.Production = value
		case "staging":
			e.Staging = value
		case "development":
			e.Development = value
		case "test":
			e.Test = value
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'production', 'staging', 'development', 'test'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (e EnvString) MarshalJSON() ([]byte, error) {
	if e.Production == nil && e.Staging == nil && e.Development == nil && e.Test == nil {
		return json.Marshal(e.Default)
	}

	m := make(map[string]interface{})
	m["default"] = e.Default
	if e.Production != nil {
		m["production"] = e.Production
	}
	if e.Staging != nil {
		m["staging"] = e.Staging
	}
	if e.Development != nil {
		m["development"] = e.Development
	}
	if e.Test != nil {
		m["test"] = e.Test
	}

	return json.Marshal(m)
}

// RegionString is a multi region wrapper for string.
type RegionString struct {
	Default *string
	USWest2 *string
	USEast1 *string
}

// Value determines the value given a region.
func (r RegionString) Value(region string) *string {
	var found *string

	switch strings.ToLower(region) {
	case "us-west-2":
		found = r.USWest2
	case "us-east-1":
		found = r.USEast1
	}

	if found != nil {
		return found
	}

	return r.Default
}

// UnmarshalJSON unmarshals either a string into Default or map of strings
func (r *RegionString) UnmarshalJSON(b []byte) error {
	var def string
	if err := json.Unmarshal(b, &def); err == nil {
		r.Default = &def
		return nil
	}

	m := make(map[string]string)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		loopSafePtr := value
		switch strings.ToLower(key) {
		case "default":
			r.Default = &loopSafePtr
		case "us-west-2":
			r.USWest2 = &loopSafePtr
		case "us-east-1":
			r.USEast1 = &loopSafePtr
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'us-west-2', 'us-east-1'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (r RegionString) MarshalJSON() ([]byte, error) {
	if r.USWest2 == nil && r.USEast1 == nil {
		return json.Marshal(r.Default)
	}

	m := make(map[string]string)
	if r.Default != nil {
		m["default"] = *r.Default
	}
	if r.USWest2 != nil {
		m["us-west-2"] = *r.USWest2
	}
	if r.USEast1 != nil {
		m["us-east-1"] = *r.USEast1
	}

	return json.Marshal(m)
}

// EnvDuration is a multi environment, region wrapper for time.Duration
type EnvDuration struct {
	Default     time.Duration
	Production  *RegionDuration
	Staging     *RegionDuration
	Development *RegionDuration
	Test        *RegionDuration
}

// Value determines the value given an environment and region.
func (e EnvDuration) Value(env, region string) time.Duration {
	var found *RegionDuration
	var value *time.Duration

	switch strings.ToLower(env) {
	case "production":
		found = e.Production
	case "staging":
		found = e.Staging
	case "development":
		found = e.Development
	case "test":
		found = e.Test
	}

	if found != nil {
		value = found.Value(region)
	}

	if value != nil {
		return *value
	}

	return e.Default
}

// UnmarshalJSON unmarshals either a time.Duration into Default or map of RegionDurations
func (e *EnvDuration) UnmarshalJSON(b []byte) error {
	var def time.Duration
	if err := json.Unmarshal(b, &def); err == nil {
		e.Default = def
		return nil
	}

	m := make(map[string]*RegionDuration)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		if value == nil {
			continue
		}
		switch strings.ToLower(key) {
		case "default":
			e.Default = *value.Default
		case "production":
			e.Production = value
		case "staging":
			e.Staging = value
		case "development":
			e.Development = value
		case "test":
			e.Test = value
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'production', 'staging', 'development', 'test'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (e EnvDuration) MarshalJSON() ([]byte, error) {
	if e.Production == nil && e.Staging == nil && e.Development == nil && e.Test == nil {
		return json.Marshal(e.Default)
	}

	m := make(map[string]interface{})
	m["default"] = e.Default
	if e.Production != nil {
		m["production"] = e.Production
	}
	if e.Staging != nil {
		m["staging"] = e.Staging
	}
	if e.Development != nil {
		m["development"] = e.Development
	}
	if e.Test != nil {
		m["test"] = e.Test
	}

	return json.Marshal(m)
}

// RegionDuration is a multi region wrapper for time.Duration.
type RegionDuration struct {
	Default *time.Duration
	USWest2 *time.Duration
	USEast1 *time.Duration
}

// Value determines the value given a region.
func (r RegionDuration) Value(region string) *time.Duration {
	var found *time.Duration

	switch strings.ToLower(region) {
	case "us-west-2":
		found = r.USWest2
	case "us-east-1":
		found = r.USEast1
	}

	if found != nil {
		return found
	}

	return r.Default
}

// UnmarshalJSON unmarshals either a time.Duration into Default or map of time.Durations
func (r *RegionDuration) UnmarshalJSON(b []byte) error {
	var def time.Duration
	if err := json.Unmarshal(b, &def); err == nil {
		r.Default = &def
		return nil
	}

	m := make(map[string]time.Duration)
	if err := json.Unmarshal(b, &m); err != nil {
		return err
	}

	for key, value := range m {
		loopSafePtr := value
		switch strings.ToLower(key) {
		case "default":
			r.Default = &loopSafePtr
		case "us-west-2":
			r.USWest2 = &loopSafePtr
		case "us-east-1":
			r.USEast1 = &loopSafePtr
		default:
			return fmt.Errorf("invalid key %q, expected to be one of 'default', 'us-west-2', 'us-east-1'.", key)
		}
	}

	return nil
}

// MarshalJSON marshals to a bool if only default is set, otherwise a map.
func (r RegionDuration) MarshalJSON() ([]byte, error) {
	if r.USWest2 == nil && r.USEast1 == nil {
		return json.Marshal(r.Default)
	}

	m := make(map[string]time.Duration)
	if r.Default != nil {
		m["default"] = *r.Default
	}
	if r.USWest2 != nil {
		m["us-west-2"] = *r.USWest2
	}
	if r.USEast1 != nil {
		m["us-east-1"] = *r.USEast1
	}

	return json.Marshal(m)
}
