package configs

import (
	"fmt"
	"io/ioutil"
	"os"
	"reflect"
	"strings"
	"time"
	"unicode"
)

func Environment() string {
	environment := os.Getenv("ENVIRONMENT")
	if environment == "" {
		environment = "development"
	}
	return environment
}

func Region() string {
	region := os.Getenv("REGION")
	if region == "" {
		region = "us-west-2"
	}
	return region
}

func EnvironmentRegion() string {
	return fmt.Sprintf("%s-%s", Environment(), Region())
}

func IsProduction() bool {
	return Environment() == "production"
}

// Config contains the configuration values for users-service
//go:generate multiconfig code.justin.tv/web/users-service/configs Config
type Config struct {
	_ struct{} `environments:"production,staging,development,test" regions:"us-west-2,us-east-1"`

	SiteDB       DB
	Reservations DB

	UserMutations KinesisStream
	Topics        Topics
	Hosts         Hosts

	LegacyAWSKey    string `secret:"web-rails/users-service/{ENVIRONMENT}/{REGION}/legacy_aws_key"`
	LegacyAWSSecret string `secret:"web-rails/users-service/{ENVIRONMENT}/{REGION}/legacy_aws_secret"`

	XraySampling float64

	Twilio Twilio

	GeoIPPath string

	CartmanKeyPath string `secret_file:"identity/cartman/{ENVIRONMENT}/ecc_public_key"`

	SQSRegion   string
	SQSEndpoint string

	NoopRailsClient bool

	STSAssumeRole string

	Workers WorkerQueues

	RollbarToken string `secret:"web-rails/users-service/{ENVIRONMENT}/{REGION}/rollbar_token"`

	Leviathan Leviathan

	Cache    Cache
	RedisTTL int

	SeedVerifyCode string

	Stats Stats

	HystrixSampling      float32
	HystrixErrorSampling float32
}

func environmentNameFromType(t reflect.Type, parents ...reflect.Type) string {
	names := make([]string, len(parents))
	for i := range parents {
		names[i] = parents[i].Name()
	}

	names = append(names, t.Name())
	for i := range names {
		names[i] = environmentNameFromString(names[i])
	}

	return strings.Join(names, "_")
}

func environmentNameFromString(name string) string {
	var env string
	// whether or not last character was upper cased
	lastUpper := true

	for _, r := range name {
		if unicode.IsUpper(r) {
			if lastUpper {
				env += string(unicode.ToUpper(r))
				continue
			}

			env += "_" + string(unicode.ToUpper(r))
			lastUpper = true
			continue
		}

		env += string(unicode.ToUpper(r))
		lastUpper = false
	}
	return env
}

type Stats struct {
	HostPort string
	App      string
}

type DB struct {
	Name         string
	User         string `secret:"sitedb:web-rails/users-service/{ENVIRONMENT}/{REGION}/db_user,reservations:web-rails/users-service/{ENVIRONMENT}/{REGION}/reservation_db_user"`
	Password     string `secret:"sitedb:web-rails/users-service/{ENVIRONMENT}/{REGION}/sitedb_password,reservations:web-rails/users-service/{ENVIRONMENT}/{REGION}/reservation_db_password"`
	PasswordFile string
	Slave        DBHost
	Master       DBHost
}

func (db DB) ResolvePassword() (string, error) {
	if db.Password == "" && db.PasswordFile != "" {
		passwordArray, err := ioutil.ReadFile(db.PasswordFile)
		if err != nil {
			return "", err
		}

		return strings.TrimSpace(string(passwordArray)), nil
	}

	return db.Password, nil
}

type DBHost struct {
	Host               string
	Port               int
	MaxOpenConns       int
	MaxIdleConns       int
	MaxQueueSize       int
	ConnAcquireTimeout time.Duration
	RequestTimeout     time.Duration
	MaxConnAge         time.Duration
	LoggerPrefix       string
}

type KinesisStream struct {
	Role       string
	Name       string
	Region     string
	RetryCount int
	RetryDelay int
}

type Cache struct {
	Primary Redis
	Backup  Redis
}

type Redis struct {
	Host           string
	Password       string `secret:"web-rails/users-service/{ENVIRONMENT}/{REGION}/redis_pass"`
	UseClusterMode bool
	ConnectTimeout int
	ReadTimeout    int
	WriteTimeout   int
	MaxConnections int
}

type Topics struct {
	SNSRegions string

	ModerationEvents      string
	RenameEvents          string
	CreationEvents        string
	MutationEvents        string
	ImageUploadEvents     string
	ChannelMutationEvents string
	EmailVerified         string
	PushyDispatched       string
	UserSoftDeleteEvents  string
	UserUndeleteEvents    string
	UserHardDeleteEvents  string
	ExpireCacheEvents     string
}

type Hosts struct {
	Follows       string
	Partnerships  string
	Spade         string
	Auditor       string
	Discovery     string
	EVS           string
	UploadService string
	Owl           string
	PubSub        string
	Rails         string
}

type Leviathan struct {
	Host  string
	Token string `secret:"web-rails/users-service/{ENVIRONMENT}/{REGION}/leviathan_token"`
}

type Twilio struct {
	Account      string `secret:"web-rails/users-service/{ENVIRONMENT}/{REGION}/twilio_account"`
	Auth         string `secret:"web-rails/users-service/{ENVIRONMENT}/{REGION}/twilio_auth"`
	PhoneNumbers string `secret:"web-rails/users-service/{ENVIRONMENT}/{REGION}/twilio_phone_numbers"`
	Disabled     bool
}

type WorkerQueues struct {
	EmailVerified WorkerQueue
	ImageUpload   WorkerQueue
	DeadLetter    WorkerQueue
	ExpireCache   WorkerQueue
}

type WorkerQueue struct {
	NumWorkers           int
	QueueName            string
	AccountNumber        string
	QueueEndpoint        string
	LogEventName         string
	Disabled             bool
	MaxVisibilityTimeout int
}
