package configs

import (
	"strings"
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/pkg/errors"
)

var (
	ErrNoSandstormConfig       = errors.New("cannot run sandstorm without a config in given region and environment")
	ErrNoRegionSandstormConfig = errors.New("cannot find sandstorm config for this region")
	ErrNoSecret                = errors.New("no secret at path")
)

type SandstormConfig struct {
	Region      string
	Environment string
	Role        string
}

type Sandstorm struct {
	enviroment string
	region     string

	manager *manager.Manager
}

func EnvRegionSandstormConfig(env, region string, configs []SandstormConfig) (*SandstormConfig, error) {
	var hasFoundConfig bool
	var hasFoundEnv bool
	var foundConfig SandstormConfig
	for _, config := range configs {
		hasFoundEnv = hasFoundEnv || config.Environment == env
		if config.Environment == env && config.Region == region {
			hasFoundConfig = true
			foundConfig = config
			break
		}
	}

	if !hasFoundConfig {
		if hasFoundEnv {
			return nil, ErrNoRegionSandstormConfig
		}
		return nil, ErrNoSandstormConfig
	}

	return &foundConfig, nil
}

func NewSandstorm(env, region string, configs []SandstormConfig) (*Sandstorm, error) {
	awsConfig := &aws.Config{Region: aws.String("us-west-2")}
	stsclient := sts.New(session.New(awsConfig))

	config, err := EnvRegionSandstormConfig(env, region, configs)
	if err != nil {
		return nil, err
	}

	// Create stscreds.AssumeRoleProvider with a 15 minute duration
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      config.Role,
		Client:       stsclient,
	}

	// Create credentials and config using our Assumed Role that we
	// will use to access the main account with
	credentials := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(credentials)

	// Finally, create a secret manager that uses the cross-account
	// config.
	sandstormManager := manager.New(manager.Config{
		AWSConfig: awsConfig,
		TableName: "sandstorm-production",
		KeyID:     "alias/sandstorm-production",
	})

	return &Sandstorm{
		enviroment: config.Environment,
		region:     config.Region,

		manager: sandstormManager,
	}, nil
}

func (s *Sandstorm) Get(path string) (string, error) {
	path = strings.Replace(path, "{ENVIRONMENT}", s.enviroment, -1)
	path = strings.Replace(path, "{REGION}", s.region, -1)

	secret, err := s.manager.Get(path)
	if err != nil {
		return "", err
	}

	if secret == nil {
		return "", ErrNoSecret
	}

	return string(secret.Plaintext), nil
}
