package configs

import (
	"context"
	"fmt"
	"io/ioutil"
	"os"
	"reflect"
	"strings"

	"code.justin.tv/chat/golibs/logx"
	"github.com/pkg/errors"
)

var (
	ErrMalformedSecretMap = errors.New("malformed secret map")
	ErrNoSecretMapMatch   = errors.New("struct with secret map used without a match")
)

type SecretFile struct {
	Path string
	File *os.File
}

type SecretFiles []SecretFile

func (secretFiles SecretFiles) Close() error {
	ctx := context.Background()

	for _, secretFile := range secretFiles {
		if secretFile.File == nil {
			continue
		}

		if err := secretFile.File.Close(); err != nil {
			logx.Warn(ctx, fmt.Sprintf("failed to close sandstorm file for %s: %s", secretFile.Path, err))
		}
		if err := os.Remove(secretFile.File.Name()); err != nil {
			logx.Error(ctx, fmt.Sprintf("failed to cleanup sandstorm file for %s: %s", secretFile.Path, err))
		}
		secretFile.File = nil
	}

	return nil
}

func ReflectTypeValue(e interface{}) (reflect.Type, reflect.Value, error) {
	t := reflect.TypeOf(e)
	v := reflect.ValueOf(e)

	if t.Kind() == reflect.Ptr {
		t = t.Elem()
		v = v.Elem()
	} else {
		return nil, reflect.Value{}, errors.New("interface must be a pointer")
	}

	return t, v, nil
}

//go:generate mockery -name SecretGetter -inpkg
type SecretGetter interface {
	Get(string) (string, error)
}

func LoadSecrets(c interface{}, m SecretGetter) (SecretFiles, error) {
	return loadSecretsWithParent(c, "", m)
}

func loadSecretsWithParent(c interface{}, parent string, m SecretGetter) (SecretFiles, error) {
	secretFiles := make(SecretFiles, 0)

	t, v, err := ReflectTypeValue(c)
	if err != nil {
		return nil, err
	}

	for i := 0; i < t.NumField(); i++ {
		sft := t.Field(i)
		sfv := v.Field(i)

		if !sfv.CanSet() {
			continue
		}

		switch sft.Type.Kind() {
		case reflect.String:
			path := sft.Tag.Get("secret")
			filePath := sft.Tag.Get("secret_file")
			if path != "" {
				if strings.Contains(path, ",") {
					originalPath := path
					pieces := strings.Split(path, ",")
					for _, piece := range pieces {
						parentSecret := strings.Split(piece, ":")
						if len(parentSecret) != 2 {
							return nil, ErrMalformedSecretMap
						}
						if strings.EqualFold(parentSecret[0], parent) {
							path = parentSecret[1]
						}
					}

					if path == originalPath {
						continue
					}
				}

				secret, err := m.Get(path)
				if err != nil {
					return nil, fmt.Errorf("failed to get sandstorm secret %s: %s", path, err)
				}

				sfv.Set(reflect.ValueOf(secret))
			} else if filePath != "" {
				secret, err := m.Get(filePath)
				if err != nil {
					return nil, fmt.Errorf("failed to get sandstorm secret %s: %s", path, err)
				}

				sf := SecretFile{
					Path: path,
				}

				sf.File, err = ioutil.TempFile("", sft.Name)
				if err != nil {
					return nil, err
				}

				_, err = sf.File.Write([]byte(secret))
				if err != nil {
					return nil, errors.Wrapf(err, "failed to write secret %s to temp file", sf.Path)
				}

				secretFiles = append(secretFiles, sf)

				sfv.Set(reflect.ValueOf(sf.File.Name()))
			}
		case reflect.Struct:
			// recurse
			files, err := loadSecretsWithParent(sfv.Addr().Interface(), sft.Name, m)
			if err != nil {
				return nil, err
			}

			secretFiles = append(secretFiles, files...)
		}

	}
	return secretFiles, nil
}
