#!/usr/bin/env bash
set -e

function log {
    echo "$(date +%Y-%m-%d:%H:%M:%S): ${1}"
}

SERVICE_PATH="/var/lib/service/users-service"
# Service needs to start and stay up for 20 seconds
CHECK_PERIOD=20
# The service _should_ restart in 40 seconds, and at max 50 seconds, however we'll set the wait time a little higher
WAIT_TIMEOUT=60

get_pid() {
  sudo /usr/bin/svstat $SERVICE_PATH | grep  "pid [0-9]\+" -o | cut -d " " -f 2
}

get_uptime() {
  sudo svstat "${SERVICE_PATH}/" | grep  ") [0-9]\+ " -o | cut -d " " -f 2
}

wait_for_pid_change() {
  NEW_PID=$(get_pid)
  SLEEP_TIME=2
  while [[ "$NEW_PID" == "$CURRENT_PID" && "$WAIT_TIMEOUT" -gt "0" ]]
  do
    log "Waiting for PID change"
    sleep $SLEEP_TIME
    NEW_PID=$(get_pid)
    TIMEOUT=$((TIMEOUT-$SLEEP_TIME))
  done

  if [[ "$NEW_PID" == "$CURRENT_PID" ]]
  then
    log "Timeout while waiting for PID change"
    exit 1;
  fi

}

monitor_for_check_period() {
  sleep $CHECK_PERIOD
  UPTIME=$(get_uptime)
  if [[ $UPTIME -lt "$CHECK_PERIOD" ]]
  then
    log "Process failed to stay up for $CHECK_PERIOD seconds"
    exit 1;
  fi
}

CURRENT_PID=$(get_pid)
log "Current PID:$CURRENT_PID"

if [ -d "$SERVICE_PATH" ]
then
    log "======== Restarting Service ========"
    sudo /usr/bin/svc -i $SERVICE_PATH
else
    log "Can't find service: $SERVICE_PATH"
    exit 1;
fi

wait_for_pid_change

log "New PID:$NEW_PID"

log "Monitoring process for $CHECK_PERIOD seconds"
monitor_for_check_period

log "Successful restart!"
exit 0
