package database

import (
	"log"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"golang.org/x/net/context"

	"code.justin.tv/chat/db"
)

const defaultStatSampleRate = 1.0

// Querier is used to query the database.
// It's a wrapper around db.DB for mocking purposes.
//go:generate mockery -name Querier
type Querier interface {
	Query(ctx context.Context, name, query string, args ...interface{}) (Rows, error)
	QueryRow(ctx context.Context, name, query string, args ...interface{}) Row
	Exec(ctx context.Context, name, query string, args ...interface{}) (db.Result, error)
	Begin(ctx context.Context, name string) (db.Tx, error)
}

// Row is a row returned by the DB.
//go:generate mockery -name Row
type Row interface {
	Scan(dest ...interface{}) error
}

//go:generate mockery -name Rows
type Rows interface {
	Close() error
	Columns() ([]string, error)
	Err() error
	Next() bool
	Scan(dest ...interface{}) error
}

type querierImpl struct {
	db db.DB
}

//go:generate mockery -name Result
type Result interface {
	LastInsertId() (int64, error)
	RowsAffected() (int64, error)
}

//go:generate mockery -name Tx
type Tx interface {
	Commit() error
	Rollback() error
	Exec(ctx context.Context, name, query string, args ...interface{}) (db.Result, error)
	Query(ctx context.Context, name, query string, args ...interface{}) (db.Rows, error)
	QueryRow(ctx context.Context, name string, query string, args ...interface{}) db.Row
}

// NewQuerier allocates and returns a Querier
func NewQuerier(db db.DB) (Querier, error) {
	return &querierImpl{
		db: db,
	}, nil
}

func (q *querierImpl) Query(ctx context.Context, name, query string, args ...interface{}) (Rows, error) {
	return q.db.Query(ctx, name, query, args...)
}

func (q *querierImpl) QueryRow(ctx context.Context, name, query string, args ...interface{}) Row {
	return q.db.QueryRow(ctx, name, query, args...)
}

func (q *querierImpl) Exec(ctx context.Context, name, query string, args ...interface{}) (db.Result, error) {
	return q.db.Exec(ctx, name, query, args...)
}

func (q *querierImpl) Begin(ctx context.Context, name string) (db.Tx, error) {
	return q.db.Begin(ctx, name)
}

// Logger is used to send database metrics to statsd
type Logger interface {
	LogDBStat(evName, queryName string, d time.Duration)
	LogRunStat(evName string)
	LogDBState(info db.DBInfo)
}

type loggerImpl struct {
	stats       statsd.Statter
	statsPrefix string
}

// NewLogger allocates and returns a Logger
func NewLogger(stats statsd.Statter, statsPrefix string) (Logger, error) {
	return &loggerImpl{
		stats:       stats,
		statsPrefix: statsPrefix,
	}, nil
}

func (l *loggerImpl) LogDBStat(evName, queryName string, d time.Duration) {
	err := l.stats.TimingDuration(l.stat("query."+queryName+"."+evName), d, defaultStatSampleRate)
	if err != nil {
		log.Printf("error while writing stats: %v", err)
	}
}

func (l *loggerImpl) LogRunStat(evName string) {
	err := l.stats.Inc(l.stat("event."+evName), 1, defaultStatSampleRate)
	if err != nil {
		log.Printf("error while writing stats: %v", err)
	}
}

func (l *loggerImpl) LogDBState(info db.DBInfo) {
	err := l.stats.Gauge(l.stat("state.open_conns_cap"), int64(info.OpenConnsCap), 1)
	if err != nil {
		log.Printf("error while writing stats: %v", err)
	}

	err = l.stats.Gauge(l.stat("state.max_open_conns"), int64(info.MaxOpenConns), 1)
	if err != nil {
		log.Printf("error while writing stats: %v", err)
	}

	err = l.stats.Gauge(l.stat("state.min_available_conns"), int64(info.MinAvailableConns), 1)
	if err != nil {
		log.Printf("error while writing stats: %v", err)
	}
}

func (l *loggerImpl) stat(stat string) string {
	return l.statsPrefix + stat
}
