#!/bin/bash -ex

function ebw() {
  docker pull docker-registry.internal.justin.tv/identity/awsebcli:latest
  docker run --rm --env-file awskeys -v $PWD:/home/aws docker-registry.internal.justin.tv/identity/awsebcli:latest "$@"
}

function eb-deploy() {
  APP=$1

  rm -f previous.txt

  ebw status $APP | sed -e '/^  Deployed/!d' -e 's/^  Deployed Version: //' > previous.txt

  rm -f failed

  ebw deploy $APP --label "$LABEL" --timeout 40 || touch failed

  if [[ -e failed ]]; then
    scripts/rollback.sh $APP;
  fi

  rm -f previous.txt
}

echo "==> ${DEPLOY_ENVIRONMENT}: PARSING PARAMETERS <=="
 
 IFS='-' read -ra PIECES <<< "$DEPLOY_ENVIRONMENT"

export REGION="${PIECES[0]}-${PIECES[1]}-${PIECES[2]}"
export ENVIRONMENT=${PIECES[3]}
export TYPE=${PIECES[4]}

export IMAGE=docker-registry.internal.justin.tv/web/users-service-${TYPE}:$GIT_COMMIT
TIMESTAMP=$(TZ=US/Pacific date -u +"%Y-%m-%dT%H:%M:%SZ")
export LABEL="${TYPE}_${TIMESTAMP:0:19}_${GIT_COMMIT:0:10}"

echo "==> ${DEPLOY_ENVIRONMENT}: CREATING DOCKERRUN.AWS.JSON <=="

cat <<EOF > Dockerrun.aws.json
{
  "AWSEBDockerrunVersion": "1",
  "Image": {
    "Name": "$IMAGE",
    "Update": "false"
  },
  "Ports": [
    {
      "ContainerPort": "8000"
    }
  ],
  "Volumes": [
    {
      "ContainerDirectory": "/var/app",
      "HostDirectory": "/var/app"
    },
    {
      "ContainerDirectory": "/usr/share",
      "HostDirectory": "/usr/share"
    },
    {
      "ContainerDirectory": "/etc/ssl/certs/ca-bundle.crt",
      "HostDirectory": "/etc/ssl/certs/ca-bundle.crt"
    }
  ],
  "Logging": "/var/log/"
}
EOF

export APP_NAME="${ENVIRONMENT}-web-users-service"
export SERVICE_NAME="${ENVIRONMENT}-web-users-service-${TYPE}"

echo "==> ${DEPLOY_ENVIRONMENT}: CREATING EB CONFIG <=="

mkdir -p .elasticbeanstalk
cat <<EOF > .elasticbeanstalk/config.yml
global:
  application_name: ${APP_NAME}
  default_region: ${REGION}
option_settings:
  - namespace: aws:elasticbeanstalk:command
    option_name: Timeout
    value: 1800

EOF

echo "==> ${DEPLOY_ENVIRONMENT}: COPYING ENV/REGION SPECIFIC CONFIGS <=="

cp -r config/${TYPE}/${ENVIRONMENT}/${REGION}/. .ebextensions/

echo "==> ${DEPLOY_ENVIRONMENT}: REMOVE OLDER APPLICATION VERSIONS <=="

ebw labs cleanup-versions --num-to-leave 10 --older-than 5 --force -v --region $REGION

echo "==> ${DEPLOY_ENVIRONMENT}: DEPLOYING <=="
eb-deploy "${SERVICE_NAME}"
