package auth

import (
	"net/http"

	"code.justin.tv/common/goauthorization"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

const (
	capabilityNameEditChannel   = "edit_channel"
	capabilityNameEditUser      = "edit_user_by_id_internal"
	capabilityNameReadUser      = "read_user_by_id"
	capabilityNameBulkReadUsers = "bulk_read_users"

	algorithm = "ES256"
	audience  = "code.justin.tv/web/users-service"
	issuer    = "code.justin.tv/web/cartman"
)

var AuthFailureError = &models.CodedError{
	ErrorValue:      "Fail to auth.",
	CodeValue:       "error_auth_failure",
	StatusCodeValue: http.StatusForbidden,
}

//go:generate mockery -name Decoder
type Decoder interface {
	AuthorizeChannelEdit(ctx context.Context, r *http.Request, editor string, channelID string) error
	AuthorizeUserEdit(ctx context.Context, r *http.Request, userID string) error
	AuthorizeUserRead(ctx context.Context, r *http.Request, userID string) error
	AuthorizeBulkUserRead(ctx context.Context, r *http.Request, userID string) error
}

type decoder struct {
	*goauthorization.Decoder
}

func NewDecoder(keyPath string) (Decoder, error) {
	d, err := goauthorization.NewDecoder(algorithm, keyPath, audience, issuer)
	if err != nil {
		return nil, err
	}

	return &decoder{d}, nil
}

func (d *decoder) AuthorizeUserRead(ctx context.Context, r *http.Request, userID string) error {
	claims := goauthorization.CapabilityClaims{
		capabilityNameReadUser: goauthorization.CapabilityClaim{
			"user_id": userID,
		},
	}

	return d.validateCapability(r, claims)
}

func (d *decoder) AuthorizeBulkUserRead(ctx context.Context, r *http.Request, userID string) error {
	claims := goauthorization.CapabilityClaims{
		capabilityNameBulkReadUsers: goauthorization.CapabilityClaim{
			"user_id": userID,
		},
	}

	return d.validateCapability(r, claims)
}

func (d *decoder) AuthorizeChannelEdit(ctx context.Context, r *http.Request, editor string, channelID string) error {
	claims := goauthorization.CapabilityClaims{
		capabilityNameEditChannel: goauthorization.CapabilityClaim{
			"channel_id": channelID,
			"user_id":    editor,
		},
	}

	return d.validateCapability(r, claims)
}

func (d *decoder) AuthorizeUserEdit(ctx context.Context, r *http.Request, userID string) error {
	claims := goauthorization.CapabilityClaims{
		capabilityNameEditUser: goauthorization.CapabilityClaim{
			"user_id": userID,
		},
	}

	return d.validateCapability(r, claims)
}

func (d *decoder) validateCapability(req *http.Request, claims goauthorization.CapabilityClaims) error {
	token, err := d.ParseToken(req)
	if err != nil {
		return err
	}

	err = d.Validate(token, claims)
	if err != nil {
		return AuthFailureError
	}

	return nil
}
