package auditor

import (
	"fmt"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/foundation/history-service/client/history"
	"code.justin.tv/web/users-service/configs"
	"github.com/afex/hystrix-go/hystrix"
	"golang.org/x/net/context"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"history_audit": {
			Timeout:               5000,
			MaxConcurrentRequests: 10000,
		},
	})

	hystrix.Configure(map[string]hystrix.CommandConfig{
		"history_search_audit": {
			Timeout:               5000,
			MaxConcurrentRequests: 10000,
		},
	})
}

func (a *auditorImpl) Search(ctx context.Context, params *history.SearchParams) (*history.SearchResults, error) {
	var result *history.SearchResults
	var err error
	err = hystrix.Do("history_search_audit", func() error {
		result, err = a.client.Search(ctx, params)
		return err
	}, nil)
	return result, err
}

func (a *auditorImpl) Audit(ctx context.Context, ev *Event) {
	if !configs.IsProduction() {
		return
	}

	if ev == nil {
		return
	}

	var changes []history.ChangeSet
	for _, change := range ev.Changes {
		hcs := history.ChangeSet{
			Attribute: change.Attribute,
			OldValue:  change.Old,
			NewValue:  change.New,
		}
		changes = append(changes, hcs)
	}

	if ev.Action == AttributeUpdate && len(changes) == 0 {
		return
	}

	if ev.UserType == "" {
		ev.UserType = DefaultUserType
	}

	createdAt := time.Now()
	if ev.CreatedAt != nil {
		createdAt = *ev.CreatedAt
	}

	auditRecord := history.Audit{
		Action:       ev.Action,
		UserType:     ev.UserType,
		UserID:       ev.UserID,
		ResourceType: ev.ResourceType,
		ResourceID:   ev.ResourceID,
		CreatedAt:    createdAt,
		Expiry:       DefaultExpiration,
		Description:  ev.Description,
		Changes:      changes,
	}

	err := a.hystrixAudit(ctx, auditRecord)
	if err != nil {
		err = a.hystrixAudit(ctx, auditRecord)
		if err != nil {
			logx.Error(ctx, "failed to add the audit record", logx.Fields{
				"Error":        fmt.Sprintf("%v", err),
				"UserType":     auditRecord.UserType,
				"UserID":       auditRecord.UserID,
				"ResourceType": auditRecord.ResourceType,
				"ResourceID":   auditRecord.ResourceID,
				"Action":       auditRecord.Action,
			})
		}
	}
}

func (a *auditorImpl) hystrixAudit(ctx context.Context, auditRecord history.Audit) error {
	return hystrix.Do("history_audit", func() error {
		return a.client.Add(ctx, auditRecord)
	}, nil)
}
