package banreport

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

type ErrorResponse struct {
	Status  int
	Message string
	Error   string
}

// BanInterface exposes methods to interact with ban reports
//go:generate mockery -name Client
type Client interface {
	ReportBan(ctx context.Context, ID string, reportingID string, banType string, isWarning bool, banReason string, reqOpts *twitchclient.ReqOpts) error
}

// NewBanClient creates a Ban client
func NewClient(host string, stats statsd.Statter) (Client, error) {
	c, err := twitchclient.NewClient(twitchclient.ClientConf{
		Host:           host,
		Stats:          stats,
		TimingXactName: "service.banreport",
	})
	if err != nil {
		return nil, err
	}

	return &banImpl{
		Client: c,
	}, nil
}

type banImpl struct {
	twitchclient.Client
}

// ReportBan reports a ban to rails
func (b *banImpl) ReportBan(ctx context.Context, ID string, reportingID string, banType string, isWarning bool, banReason string, reqOpts *twitchclient.ReqOpts) error {
	endpoint := "/api/internal/ban_report/"

	params := struct {
		ID          string `json:"id"`
		ReportingID string `json:"reporting_id"`
		Type        string `json:"type"`
		Warn        bool   `json:"warn"`
		Reason      string `json:"reason"`
	}{
		ID,
		reportingID,
		banType,
		isWarning,
		banReason,
	}

	bodyJson, err := json.Marshal(params)
	if err != nil {
		return err
	}

	body := bytes.NewBuffer(bodyJson)

	req, err := b.NewRequest("POST", endpoint, body)
	if err != nil {
		return err
	}
	req.Header.Set("Content-Type", "application/json")

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "ban_report",
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()
	if err != nil {
		return err
	}

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}
	return nil
}

// handleFailedRequest is a generic error parser / formatter for all ban report endpoints
func (b *banImpl) handleFailedRequest(resp *http.Response) error {
	var errResp ErrorResponse
	err := json.NewDecoder(resp.Body).Decode(&errResp)
	if err != nil {
		return fmt.Errorf("Failed ban report request. StatusCode=%v Unable to read response body (%v)", resp.StatusCode, err)
	}

	return fmt.Errorf(`Failed ban report request. StatusCode=%v Message=%v`, resp.StatusCode, errResp.Message)
}
