package cacherwrapper

import (
	"fmt"
	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/models"
	"github.com/afex/hystrix-go/hystrix"
	"golang.org/x/net/context"
)

const (
	hystrixSingleGetTimeout       = 700
	hystrixBulkGetTimeout         = 700
	hystrixExpireTimeout          = 1000
	hystrixBulkSetTimeout         = 1000
	hystrixSingleSetTimeout       = 1000
	hystrixSingleGetMaxConcurrent = 1500
	hystrixBulkGetMaxConcurrent   = 3000
	hystrixExpireMaxConcurrent    = 300
	hystrixBulkSetMaxConcurrent   = 500
	hystrixSingleSetMaxConcurrent = 500
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"Cache.GetProperties": {
			Timeout:               hystrixSingleGetTimeout,
			MaxConcurrentRequests: hystrixSingleGetMaxConcurrent,
		},
		"Cache.ExpireProperties": {
			Timeout:               hystrixExpireTimeout,
			MaxConcurrentRequests: hystrixExpireMaxConcurrent,
		},
		"Cache.CacheProperties": {
			Timeout:               hystrixSingleSetTimeout,
			MaxConcurrentRequests: hystrixSingleSetMaxConcurrent,
		},
		"Cache.BulkGetProperties": {
			Timeout:               hystrixBulkGetTimeout,
			MaxConcurrentRequests: hystrixBulkGetMaxConcurrent,
		},
		"Cache.BulkSetProperties": {
			Timeout:               hystrixBulkSetTimeout,
			MaxConcurrentRequests: hystrixBulkSetMaxConcurrent,
		},
	})
}

type CacherWrapperImpl struct {
	backend.Cacher
	key       string
	cacheType string
}

func New(cacher backend.Cacher, key, cacheType string) *CacherWrapperImpl {
	return &CacherWrapperImpl{cacher, key, cacheType}
}

func handleError(err error) error {
	if err == backend.ErrCacheMiss {
		return nil
	} else {
		return err
	}
}

func (c *CacherWrapperImpl) GetProperties(ctx context.Context, field string, key string, e interface{}) error {
	misses := 0
	start := time.Now()
	var err error
	err = hystrix.Do("Cache.GetProperties", func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = errx.New(fmt.Sprintf("Cache.GetProperties circuit panic=%v", p))
			}
		}()

		cacheErr := c.Cacher.GetProperties(ctx, field, key, e)
		if cacheErr != nil {
			misses = 1
		}
		return handleError(cacheErr)
	}, nil)

	err = c.collectMetrics(ctx, start, "Get", misses, 1-misses, err)
	if err != nil {
		return err
	} else if misses == 1 {
		return backend.ErrCacheMiss
	} else {
		return nil
	}
}

func (c *CacherWrapperImpl) ExpireProperties(ctx context.Context, prop models.Cacheable) error {
	start := time.Now()
	var err error
	err = hystrix.Do("Cache.ExpireProperties", func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = errx.New(fmt.Sprintf("Cache.ExpireProperties circuit panic=%v", p))
			}
		}()

		cacheErr := c.Cacher.ExpireProperties(ctx, prop)
		return handleError(cacheErr)
	}, nil)
	return c.collectMetrics(ctx, start, "Expire", 0, 0, err)
}

func (c *CacherWrapperImpl) CacheProperties(ctx context.Context, overwrite bool, props models.Cacheable) error {
	start := time.Now()
	var err error
	err = hystrix.Do("Cache.CacheProperties", func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = errx.New(fmt.Sprintf("Cache.CacheProperties circuit panic=%v", p))
			}
		}()

		cacheErr := c.Cacher.CacheProperties(ctx, overwrite, props)
		return handleError(cacheErr)
	}, nil)
	return c.collectMetrics(ctx, start, "Set", 0, 0, err)
}

func (c *CacherWrapperImpl) BulkGetProperties(ctx context.Context, field string, key []string, e interface{}) ([]int, error) {
	missings := []int{}
	var misses, hits int
	start := time.Now()
	var err error
	err = hystrix.Do("Cache.BulkGetProperties", func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = errx.New(fmt.Sprintf("Cache.BulkGetProperties circuit panic=%v", p))
			}
		}()

		cacheMissings, cacheErr := c.Cacher.BulkGetProperties(ctx, field, key, e)
		for _, m := range cacheMissings {
			missings = append(missings, m)
		}

		if cacheErr != nil && key != nil {
			misses = len(key)
		} else {
			misses = len(missings)
		}

		if key != nil {
			hits = len(key) - misses
		}
		return handleError(cacheErr)
	}, nil)

	return missings, c.collectMetrics(ctx, start, "BulkGet", misses, hits, err)
}

func (c *CacherWrapperImpl) BulkSetProperties(ctx context.Context, overwrite bool, iter models.CacheableIterator) error {
	start := time.Now()
	var err error
	err = hystrix.Do("Cache.BulkSetProperties", func() (err error) {
		defer func() {
			if p := recover(); p != nil {
				err = errx.New(fmt.Sprintf("Cache.BulkSetProperties circuit panic=%v", p))
			}
		}()

		cacheErr := c.Cacher.BulkSetProperties(ctx, overwrite, iter)
		return handleError(cacheErr)
	}, nil)
	return c.collectMetrics(ctx, start, "BulkSet", 0, 0, err)
}

func (c *CacherWrapperImpl) collectMetrics(ctx context.Context, start time.Time, name string, misses, hits int, err error) error {
	duration := time.Since(start)
	if metricErr := config.ClusterStatsd().TimingDuration(fmt.Sprintf("%s.%s.%s.duration", c.cacheType, c.key, name), duration, 0.1); metricErr != nil {
		logx.Warn(ctx, fmt.Sprintf("error incrementing cache metrics %s duration: %s", name, metricErr))
	}

	if err != nil && err != backend.ErrCacheMiss {
		if metricErr := config.ClusterStatsd().Inc(fmt.Sprintf("%s.%s.%s.failure", c.cacheType, c.key, name), 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing cache metrics %s failure: %s", name, metricErr))
		}
	} else {
		if metricErr := config.ClusterStatsd().Inc(fmt.Sprintf("%s.%s.%s.success", c.cacheType, c.key, name), 1, 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing cache metrics %s success: %s", name, metricErr))
		}
	}

	if misses != 0 {
		if metricErr := config.ClusterStatsd().Inc(fmt.Sprintf("%s.%s.%s.miss", c.cacheType, c.key, name), int64(misses), 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing cache metrics %s miss: %s", name, metricErr))
		}
	}

	if hits != 0 {
		if metricErr := config.ClusterStatsd().Inc(fmt.Sprintf("%s.%s.%s.hit", c.cacheType, c.key, name), int64(hits), 0.1); metricErr != nil {
			logx.Warn(ctx, fmt.Sprintf("error incrementing cache metrics %s hit: %s", name, metricErr))
		}
	}

	if err != nil && err != backend.ErrCacheMiss {
		return err
	} else {
		return nil
	}
}

func (c *CacherWrapperImpl) Close() error {
	return c.Cacher.Close()
}
