package memcacher_test

import (
	"testing"

	"code.justin.tv/foundation/gomemcache/memcache"

	"encoding/json"

	"code.justin.tv/web/users-service/internal/clients/cache/memcacher"
	"code.justin.tv/web/users-service/internal/clients/cache/memcacher/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type ExpireTest struct {
	suite.Suite
	mockMemcached *mocks.Memcached
	cb            *memcacher.CacherImpl
	items         []*memcache.Item
}

func (suite *ExpireTest) SetupTest() {
	suite.mockMemcached = &mocks.Memcached{}
	suite.cb = &memcacher.CacherImpl{
		Memcache:   suite.mockMemcached,
		Key:        "user-props-string",
		Expiration: 30000,
	}
	valueBytes, err := json.Marshal(Users[AdminUserID])
	assert.NoError(suite.T(), err)
	for _, pair := range Users[AdminUserID].CachePairs() {
		suite.items = append(suite.items, &memcache.Item{
			Expiration: suite.cb.Expiration,
			Key:        suite.cb.PropertiesByFieldCacheKey(pair.Key, pair.Value),
			Value:      valueBytes,
			Flags:      0,
		})
	}
}

func (suite *ExpireTest) TestExpireProperty() {
	for _, item := range suite.items {
		suite.mockMemcached.On("Delete", item.Key).Return(nil)
	}
	err := suite.cb.ExpireProperties(context.Background(), Users[AdminUserID])
	assert.NoError(suite.T(), err)
	suite.mockMemcached.AssertExpectations(suite.T())
}

func (suite *ExpireTest) TestExpirePropertyWithOneCacheMiss() {
	for i, item := range suite.items {
		if i == 0 {
			suite.mockMemcached.On("Delete", item.Key).Return(memcache.ErrCacheMiss)
		} else {
			suite.mockMemcached.On("Delete", item.Key).Return(nil)
		}
	}

	err := suite.cb.ExpireProperties(context.Background(), Users[AdminUserID])
	assert.NoError(suite.T(), err)
	suite.mockMemcached.AssertExpectations(suite.T())
}

func (suite *ExpireTest) TestExpirePropertyWithTwoCacheMiss() {
	for i, item := range suite.items {
		if i <= 1 {
			suite.mockMemcached.On("Delete", item.Key).Return(memcache.ErrCacheMiss)
		} else {
			suite.mockMemcached.On("Delete", item.Key).Return(nil)
		}
	}

	err := suite.cb.ExpireProperties(context.Background(), Users[AdminUserID])
	assert.NoError(suite.T(), err)
	suite.mockMemcached.AssertExpectations(suite.T())
}

func (suite *ExpireTest) TestExpirePropertyWithAllCacheMiss() {
	for _, item := range suite.items {
		suite.mockMemcached.On("Delete", item.Key).Return(memcache.ErrCacheMiss)
	}

	err := suite.cb.ExpireProperties(context.Background(), Users[AdminUserID])
	assert.NoError(suite.T(), err)
	suite.mockMemcached.AssertExpectations(suite.T())
}

func (suite *ExpireTest) TestExpireExistingNilUserProperties() {
	err := suite.cb.ExpireProperties(context.Background(), nil)
	assert.NoError(suite.T(), err)
}

func TestExpireSuite(t *testing.T) {
	suite.Run(t, new(ExpireTest))
}
