package memcacher_test

import (
	"testing"

	"encoding/json"

	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/web/users-service/internal/clients/cache/memcacher"
	"code.justin.tv/web/users-service/internal/clients/cache/memcacher/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type GetTest struct {
	suite.Suite
	mockMemcached *mocks.Memcached
	cb            *memcacher.CacherImpl
}

func (suite *GetTest) SetupTest() {
	suite.mockMemcached = &mocks.Memcached{}
	suite.cb = &memcacher.CacherImpl{
		Memcache:   suite.mockMemcached,
		Key:        "user-props-string",
		Expiration: 30000,
	}
}

func (suite *GetTest) TestGetUserProperties() {
	s, err := json.Marshal(Users[ToBanUserID])
	item := &memcache.Item{
		Value: s,
	}
	suite.mockMemcached.On("Get", suite.cb.PropertiesByFieldCacheKey("u.id", ToBanUserID)).Return(item, nil)
	up := &models.Properties{}
	err = suite.cb.GetProperties(context.Background(), "u.id", ToBanUserID, up)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), up, Users[ToBanUserID])
}

func (suite *GetTest) TestGetUsersProperties() {
	keys := []string{AdminUserID, ToBanUserID}
	cacheKeys := make([]string, len(keys))
	for i, key := range keys {
		cacheKeys[i] = suite.cb.PropertiesByFieldCacheKey("u.id", key)
	}

	s, err := json.Marshal(Users[ToBanUserID])
	itemToBan := &memcache.Item{
		Value: s,
	}

	s, err = json.Marshal(Users[AdminUserID])
	itemAdmin := &memcache.Item{
		Value: s,
	}
	value := map[string]*memcache.Item{
		suite.cb.PropertiesByFieldCacheKey("u.id", ToBanUserID): itemToBan,
		suite.cb.PropertiesByFieldCacheKey("u.id", AdminUserID): itemAdmin,
	}
	suite.mockMemcached.On("GetMulti", cacheKeys).Return(value, nil)
	var up []models.Properties
	missingKeys, err := suite.cb.BulkGetProperties(context.Background(), "u.id", keys, &up)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), missingKeys, []int{})
	assert.Equal(suite.T(), up, []models.Properties{*Users[AdminUserID], *Users[ToBanUserID]})
}

func (suite *GetTest) TestGetUsersPropertiesMixed() {
	keys := []string{AdminUserID, ToBanUserID}
	cacheKeys := make([]string, len(keys))
	for i, key := range keys {
		cacheKeys[i] = suite.cb.PropertiesByFieldCacheKey("u.id", key)
	}

	s, err := json.Marshal(Users[ToBanUserID])
	itemToBan := &memcache.Item{
		Value: s,
	}

	value := map[string]*memcache.Item{
		suite.cb.PropertiesByFieldCacheKey("u.id", ToBanUserID): itemToBan,
	}
	suite.mockMemcached.On("GetMulti", cacheKeys).Return(value, nil)
	var up []models.Properties
	missingKeys, err := suite.cb.BulkGetProperties(context.Background(), "u.id", keys, &up)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), missingKeys, []int{0})
	assert.Equal(suite.T(), up, []models.Properties{*Users[ToBanUserID]})
}

func TestGetSuite(t *testing.T) {
	suite.Run(t, new(GetTest))
}
