package memcacher

import (
	"encoding/json"

	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"
)

func (c *CacherImpl) BulkSetProperties(ctx context.Context, overwrite bool, iter models.CacheableIterator) error {
	var props []models.Cacheable
	err := iter.Each(func(p models.Cacheable) error {
		props = append(props, p)
		return nil
	})

	var g errgroup.Group
	for _, prop := range props {
		prop := prop
		g.Go(func() error {
			return c.CacheProperties(ctx, overwrite, prop)
		})
	}

	err = g.Wait()
	return err
}

func (c *CacherImpl) CacheProperties(ctx context.Context, overwrite bool, prop models.Cacheable) (err error) {
	valueBytes, err := json.Marshal(prop)
	if err != nil {
		return err
	}

	var g errgroup.Group
	for _, pair := range prop.CachePairs() {
		item := &memcache.Item{
			Expiration: c.Expiration,
			Key:        c.PropertiesByFieldCacheKey(pair.Key, pair.Value),
			Value:      valueBytes,
			Flags:      0,
		}
		if overwrite {
			g.Go(func() error {
				return c.Memcache.Set(item)
			})
		} else {
			g.Go(func() error {
				return c.addItem(item)
			})
		}
	}
	err = g.Wait()
	return err
}

func (c *CacherImpl) addItem(item *memcache.Item) error {
	err := c.Memcache.Add(item)
	if err == memcache.ErrNotStored {
		err = nil
	}
	return err
}
