package memcacher_test

import (
	"testing"

	"encoding/json"

	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/web/users-service/internal/clients/cache/memcacher"
	"code.justin.tv/web/users-service/internal/clients/cache/memcacher/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type SetTest struct {
	suite.Suite
	mockMemcached *mocks.Memcached
	cb            *memcacher.CacherImpl
	items         []*memcache.Item
}

func (suite *SetTest) SetupTest() {
	suite.mockMemcached = &mocks.Memcached{}
	suite.cb = &memcacher.CacherImpl{
		Memcache:   suite.mockMemcached,
		Key:        "user-props-string",
		Expiration: 30000,
	}
	valueBytes, err := json.Marshal(Users[AdminUserID])
	assert.NoError(suite.T(), err)
	for _, pair := range Users[AdminUserID].CachePairs() {
		suite.items = append(suite.items, &memcache.Item{
			Expiration: suite.cb.Expiration,
			Key:        suite.cb.PropertiesByFieldCacheKey(pair.Key, pair.Value),
			Value:      valueBytes,
			Flags:      0,
		})
	}
}

func (suite *SetTest) TestCachePropertiesWithOverwrite() {
	for _, item := range suite.items {
		suite.mockMemcached.On("Set", item).Return(nil)
	}
	err := suite.cb.CacheProperties(context.Background(), true, *Users[AdminUserID])
	assert.NoError(suite.T(), err)
}

func (suite *SetTest) TestCachePropertiesWithoutOverwrite() {
	for _, item := range suite.items {
		suite.mockMemcached.On("Add", item).Return(nil)
	}

	err := suite.cb.CacheProperties(context.Background(), false, *Users[AdminUserID])
	assert.NoError(suite.T(), err)
}

func (suite *SetTest) TestCacheMultiPropertiesWithOverwrite() {
	valueBytes, err := json.Marshal(Users[ToBanUserID])
	assert.NoError(suite.T(), err)
	for _, pair := range Users[ToBanUserID].CachePairs() {
		suite.items = append(suite.items, &memcache.Item{
			Expiration: suite.cb.Expiration,
			Key:        suite.cb.PropertiesByFieldCacheKey(pair.Key, pair.Value),
			Value:      valueBytes,
			Flags:      0,
		})
	}

	for _, item := range suite.items {
		suite.mockMemcached.On("Set", item).Return(nil)
	}
	err = suite.cb.BulkSetProperties(context.Background(), true, models.PropertiesIterator([]models.Properties{*Users[AdminUserID], *Users[ToBanUserID]}))
	assert.NoError(suite.T(), err)
	suite.mockMemcached.AssertExpectations(suite.T())
}

func (suite *SetTest) TestCacheMultiPropertiesWithoutOverwrite() {
	valueBytes, err := json.Marshal(Users[ToBanUserID])
	assert.NoError(suite.T(), err)
	for _, pair := range Users[ToBanUserID].CachePairs() {
		suite.items = append(suite.items, &memcache.Item{
			Expiration: suite.cb.Expiration,
			Key:        suite.cb.PropertiesByFieldCacheKey(pair.Key, pair.Value),
			Value:      valueBytes,
			Flags:      0,
		})
	}

	for _, item := range suite.items {
		suite.mockMemcached.On("Add", item).Return(nil)
	}

	err = suite.cb.BulkSetProperties(context.Background(), false, models.PropertiesIterator([]models.Properties{*Users[AdminUserID], *Users[ToBanUserID]}))
	assert.NoError(suite.T(), err)
	suite.mockMemcached.AssertExpectations(suite.T())
}

func (suite *SetTest) TestCachePropertiesWithNotStoredError() {
	valueBytes, err := json.Marshal(Users[ToBanUserID])
	assert.NoError(suite.T(), err)
	for _, pair := range Users[ToBanUserID].CachePairs() {
		suite.items = append(suite.items, &memcache.Item{
			Expiration: suite.cb.Expiration,
			Key:        suite.cb.PropertiesByFieldCacheKey(pair.Key, pair.Value),
			Value:      valueBytes,
			Flags:      0,
		})
	}

	for i, item := range suite.items {
		if i != 0 {
			suite.mockMemcached.On("Add", item).Return(nil)
		} else {
			suite.mockMemcached.On("Add", item).Return(memcache.ErrNotStored)
		}
	}

	err = suite.cb.BulkSetProperties(context.Background(), false, models.PropertiesIterator([]models.Properties{*Users[AdminUserID], *Users[ToBanUserID]}))
	assert.NoError(suite.T(), err)
	suite.mockMemcached.AssertExpectations(suite.T())
}

func (suite *SetTest) TestCachePropertiesWithUnexpctedError() {
	valueBytes, err := json.Marshal(Users[ToBanUserID])
	assert.NoError(suite.T(), err)
	for _, pair := range Users[ToBanUserID].CachePairs() {
		suite.items = append(suite.items, &memcache.Item{
			Expiration: suite.cb.Expiration,
			Key:        suite.cb.PropertiesByFieldCacheKey(pair.Key, pair.Value),
			Value:      valueBytes,
			Flags:      0,
		})
	}

	for i, item := range suite.items {
		if i != 0 {
			suite.mockMemcached.On("Add", item).Return(nil)
		} else {
			suite.mockMemcached.On("Add", item).Return(errors.New("Something unexpected."))
		}
	}

	err = suite.cb.BulkSetProperties(context.Background(), false, models.PropertiesIterator([]models.Properties{*Users[AdminUserID], *Users[ToBanUserID]}))
	assert.NotNil(suite.T(), err)
	suite.mockMemcached.AssertExpectations(suite.T())
}

func TestSetTest(t *testing.T) {
	suite.Run(t, new(SetTest))
}
