package rediscacher

import (
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/internal/clients/sns"
	"golang.org/x/net/context"

	"fmt"
)

type Marshaler interface {
	Marshal(e interface{}) ([]byte, error)
	Unmarshal(b []byte, e interface{}) error
}

type Redis interface {
	Get(ctx context.Context, cacheKey string) (string, bool, error)
	MGet(ctx context.Context, cacheKey ...string) ([]string, error)
	Del(ctx context.Context, keys ...string) (int, error)
	MSetWithTTL(ctx context.Context, keys []string, values []string, ttl time.Duration) error
	MSafeSetWithTTL(ctx context.Context, keys []string, values []string, ttl time.Duration) ([]bool, error)
	Close() error
}

//go:generate mockery -name LocalExpirationCacher
type LocalExpirationCacher interface {
	ExpireLocalCaches(ctx context.Context, keys ...string) error
}

type Flags struct {
}

type CacherImpl struct {
	redis       Redis
	redisBackup Redis
	cacheTTLMS  time.Duration
	key         string
	marshaler   Marshaler
	sns         sns.Publisher
	flags       Flags
}

func NewLocalExpirationCacher(redis, redisBackup Redis) *CacherImpl {
	return &CacherImpl{
		redis:       redis,
		redisBackup: redisBackup,
	}
}

func New(redis, redisBackup Redis, cacheTTLMS time.Duration, key string, sns sns.Publisher, flags *Flags) *CacherImpl {
	return NewMarshaler(redis, redisBackup, cacheTTLMS, key, &JSONMarshaler{}, sns, flags)
}

func NewMarshaler(redis, redisBackup Redis, cacheTTLMS time.Duration, key string, marshaler Marshaler, sns sns.Publisher, flags *Flags) *CacherImpl {
	var f Flags
	if flags != nil {
		f = *flags
	}

	return &CacherImpl{
		redis:       redis,
		redisBackup: redisBackup,
		cacheTTLMS:  cacheTTLMS,
		key:         key,
		marshaler:   marshaler,
		sns:         sns,
		flags:       f,
	}
}

func (c *CacherImpl) PropertiesByFieldCacheKey(field, identifier string) string {
	return fmt.Sprintf("%s:%s:%s", c.key, field, identifier)
}

func (c *CacherImpl) Close() error {
	if err := c.redisBackup.Close(); err != nil {
		logx.Error(context.Background(), err)
	}

	return c.redis.Close()
}
