package rediscacher

import (
	"fmt"
	"strings"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

func (c *CacherImpl) ExpireProperties(ctx context.Context, prop models.Cacheable) error {
	if prop == nil {
		return nil
	}

	cacheKeys := []string{}
	for _, pair := range prop.CachePairs() {
		cacheKeys = append(cacheKeys, c.PropertiesByFieldCacheKey(pair.Key, pair.Value))
	}

	return c.expire(ctx, cacheKeys...)
}

// This function is used by the backend to invalidate the entries in
// local redis caches and subsequesntly publish a CacheExpirationEvent
// to update caches in other regions
func (c *CacherImpl) expire(ctx context.Context, cacheKeys ...string) error {
	e := c.ExpireLocalCaches(ctx, cacheKeys...)

	bgCtx := backend.DetachContext(ctx)
	// Invoke go routine to expire event in the remote cache instances
	// This is to ensure that the caching data across multiple regions are in sync
	go func() {
		keys := strings.Join(cacheKeys, ",")
		ecEvent := models.SNSExpireCacheEvent{
			Keys:      keys,
			Timestamp: time.Now(),
		}
		err := c.sns.PublishExpireCache(bgCtx, ecEvent)
		if err != nil {
			logx.Error(bgCtx, fmt.Errorf("failed to publish cache expiration notification: %v", err))
		}
	}()

	return e
}

func (c *CacherImpl) ExpireLocalCaches(ctx context.Context, cacheKeys ...string) error {
	_, e := c.redis.Del(ctx, cacheKeys...)
	primaryOperation(ctx, "expire", e)

	// Invoke go routine on backup (To replicate primary behavior)
	bgCtx := backend.DetachContext(ctx)
	go func() {
		_, e1 := c.redisBackup.Del(bgCtx, cacheKeys...)
		replicateOperation(bgCtx, "expire", e1)
	}()
	return e
}
