package rediscacher

import (
	"fmt"
	"strings"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/web/users-service/backend"
	"golang.org/x/net/context"
)

func (c *CacherImpl) GetProperties(ctx context.Context, field, key string, e interface{}) error {
	cacheKey := c.PropertiesByFieldCacheKey(field, key)
	result := c.getProperties(ctx, cacheKey)

	if !result.ok {
		return backend.ErrCacheMiss
	}

	if result.value == "" {
		return backend.ErrCacheMiss
	}

	err := c.marshaler.Unmarshal([]byte(result.value), e)
	if err != nil {
		c.unmarshalErrExpire(ctx, cacheKey)
		return backend.ErrCacheMiss
	}
	return nil
}

type getResponse struct {
	value string
	ok    bool
	err   error
	layer string
}

func (c *CacherImpl) getProperties(ctx context.Context, cacheKey string) getResponse {
	op := func(ctx context.Context, layer string, redis Redis) getResponse {
		cacheResult, ok, err := redis.Get(ctx, cacheKey)
		layerOperation(ctx, layer, "get", err)
		return getResponse{
			value: cacheResult,
			ok:    ok,
			err:   err,
			layer: layer,
		}
	}

	result := op(ctx, "primary", c.redis)
	// if the first result we get is an error, wait for the other one to respond
	if result.err != nil {
		result = op(ctx, "backup", c.redisBackup)
	} else {
		go op(backend.DetachContext(ctx), "replicate", c.redisBackup)
	}

	if result.err == nil {
		metricUsed(result.layer, "get")
	}

	return result
}

type bulkGetResponse struct {
	value []string
	err   error
	layer string
}

func (c *CacherImpl) bulkGetProperties(ctx context.Context, cacheKeys ...string) bulkGetResponse {
	op := func(ctx context.Context, layer string, redis Redis) bulkGetResponse {
		cacheResults, err := redis.MGet(ctx, cacheKeys...)
		layerOperation(ctx, layer, "bulkget", err)
		return bulkGetResponse{
			value: cacheResults,
			err:   err,
			layer: layer,
		}
	}

	result := op(ctx, "primary", c.redis)
	if result.err != nil {
		result = op(ctx, "backup", c.redisBackup)
	} else {
		go op(backend.DetachContext(ctx), "replicate", c.redisBackup)
	}

	if result.err == nil {
		metricUsed(result.layer, "bulkget")
	}

	return result
}

func (c *CacherImpl) bulkToSingleGet(ctx context.Context, cacheKey string) ([]string, error) {
	result := c.getProperties(ctx, cacheKey)
	if result.err != nil {
		return nil, result.err
	}
	cacheResults := []string{result.value}
	return cacheResults, nil
}

func (c *CacherImpl) BulkGetProperties(ctx context.Context, field string, keys []string, e interface{}) ([]int, error) {
	cacheKeys := make([]string, len(keys))
	for i, key := range keys {
		cacheKeys[i] = c.PropertiesByFieldCacheKey(field, key)
	}
	var cacheResults []string
	var err error
	if len(keys) == 1 {
		cacheResults, err = c.bulkToSingleGet(ctx, cacheKeys[0])
	} else {
		result := c.bulkGetProperties(ctx, cacheKeys...)
		cacheResults = result.value
		err = result.err
	}

	if err != nil {
		return nil, err
	}

	misses, deletes, err := ccUnmarshalResults(c.marshaler, cacheResults, e)

	var deleteKeys []string
	for _, delete := range deletes {
		deleteKeys = append(deleteKeys, cacheKeys[delete])
	}

	if deleteKeys != nil {
		c.unmarshalErrExpire(ctx, deleteKeys...)
	}

	return misses, err
}

func (c *CacherImpl) unmarshalErrExpire(ctx context.Context, cacheKeys ...string) {
	if metricErr := config.ClusterStatsd().Inc("redis."+c.key+".unmarshal_error", int64(len(cacheKeys)), 0.1); metricErr != nil {
		logx.Error(ctx, fmt.Errorf("failed to emit redis unmarshal: %v", metricErr))
	}
	if err := c.expire(ctx, cacheKeys...); err != nil {
		logx.Error(ctx, fmt.Errorf("fail to clear redis cache: %v", err))
	}
}

// Note: This only works with the json Marshaler
func bulkUnmarshalResults(marshaler Marshaler, cacheResults []string, e interface{}) ([]int, []int, error) {
	var hits []string
	var misses []int
	for i, cacheResult := range cacheResults {
		if cacheResult == "" {
			misses = append(misses, i)
			continue
		}

		hits = append(hits, cacheResult)
	}

	commaSeparated := strings.Join(hits, ",")
	jsonStr := "[" + commaSeparated + "]"
	err := marshaler.Unmarshal([]byte(jsonStr), e)
	if err != nil {
		misses := make([]int, len(cacheResults))
		for i := range cacheResults {
			misses[i] = i
		}
		return misses, misses, err
	}
	return misses, nil, nil
}
