package rediscacher

import (
	"context"
	"encoding/json"
	"testing"
	"time"

	"code.justin.tv/chat/redis"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	. "code.justin.tv/web/users-service/internal/testutils"
)

type GetTest struct {
	suite.Suite
	mockRedis *redis.MockRedis
	cb        *CacherImpl
}

func (suite *GetTest) SetupTest() {
	suite.mockRedis = &redis.MockRedis{}
	suite.cb = &CacherImpl{
		redis:       suite.mockRedis,
		redisBackup: suite.mockRedis,
		cacheTTLMS:  time.Minute,
		key:         "user-props-string",
		marshaler:   &JSONMarshaler{},
	}
}

func (suite *GetTest) TestGetUserProperties() {
	s, err := json.Marshal(Users[ToBanUserID])
	suite.mockRedis.On("Get", mock.Anything, mock.Anything).Return(string(s), true, nil).Times(2)
	up := &models.Properties{}
	err = suite.cb.GetProperties(context.Background(), "u.id", ToBanUserID, up)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), Users[ToBanUserID], up)
}

func TestGetSuite(t *testing.T) {
	suite.Run(t, new(GetTest))
}

func (suite *GetTest) TestGetUsersProperties() {
	s, err := json.Marshal(Users[ToBanUserID])
	suite.mockRedis.On("MGet", mock.Anything, mock.Anything).Return([]string{"", string(s)}, nil).Times(2)
	var up []models.Properties
	missingKeys, err := suite.cb.BulkGetProperties(context.Background(), "u.id", []string{AdminUserID, ToBanUserID}, &up)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), missingKeys, []int{0})
	assert.Equal(suite.T(), []models.Properties{*Users[ToBanUserID]}, up)
}

func (suite *GetTest) TestGetUsersPropertiesSingleUser() {
	s, err := json.Marshal(Users[ToBanUserID])
	suite.mockRedis.On("Get", mock.Anything, mock.Anything).Return(string(s), true, nil)
	suite.mockRedis.On("MGet", mock.Anything, mock.Anything).Return([]string{string(s)}, nil)
	var up []models.Properties
	missingKeys, err := suite.cb.BulkGetProperties(context.Background(), "u.id", []string{ToBanUserID}, &up)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), missingKeys, []int(nil))
	assert.Equal(suite.T(), []models.Properties{*Users[ToBanUserID]}, up)
}
