package rediscacher

import (
	"fmt"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/web/users-service/internal/utils"
	"golang.org/x/net/context"
)

func replicateOperation(ctx context.Context, operation string, err error) {
	layerOperation(ctx, "replicate", operation, err)
}

func primaryOperation(ctx context.Context, operation string, err error) {
	layerOperation(ctx, "primary", operation, err)
}

func backupOperation(ctx context.Context, operation string, err error) {
	layerOperation(ctx, "backup", operation, err)
}

func layerOperation(ctx context.Context, layer, operation string, err error) {
	metricErr(layer, operation, err)
	logErr(ctx, layer, operation, err)
}

func metricErr(layer, operation string, err error) {
	success := "success"
	if utils.IsContextCanceled(err) {
		success = "canceled"
	} else if utils.IsTimeoutErr(err) {
		success = "timeout"
	} else if err != nil {
		success = "failure"
	}

	metric(layer, operation, success)
}

func metricUsed(layer, operation string) {
	metric(layer, operation, "used")
}

func metric(layer, operation, message string) {
	name := fmt.Sprintf("dualcache.%s.%s.%s", layer, operation, message)
	if err := config.ClusterStatsd().Inc(name, 1, 0.1); err != nil {
		logx.Warn(context.TODO(), fmt.Sprintf("failed to log redis primary/backup stat: %s", err))
	}
}

func logErr(ctx context.Context, layer, operation string, err error) {
	if err == nil {
		return
	}

	if utils.IsHystrixErr(err) || utils.IsContextCanceled(err) || utils.IsTimeoutErr(err) {
		// don't log
		return
	}

	if layer == "primary" {
		logx.Error(ctx, err, logx.Fields{
			"cache_operation": operation,
			"cache_layer":     layer,
		})
		return
	}
	logx.Warn(ctx, err, logx.Fields{
		"cache_operation": operation,
		"cache_layer":     layer,
	})
}
