package rediscacher

import (
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/models"
	"golang.org/x/net/context"
)

func (c *CacherImpl) CacheProperties(ctx context.Context, overwrite bool, prop models.Cacheable) error {
	missingCacheKeys := []string{}
	missingValues := []string{}
	marshalled, err := c.marshaler.Marshal(&prop)
	if err != nil {
		return err
	}

	for _, pair := range prop.CachePairs() {
		missingCacheKeys = append(missingCacheKeys, c.PropertiesByFieldCacheKey(pair.Key, pair.Value))
		missingValues = append(missingValues, string(marshalled))
	}

	return c.cacheProperties(ctx, overwrite, missingCacheKeys, missingValues)
}

func (c *CacherImpl) BulkSetProperties(ctx context.Context, overwrite bool, iter models.CacheableIterator) error {
	missingCacheKeys := []string{}
	missingValues := []string{}

	if err := iter.Each(func(p models.Cacheable) error {
		marshalled, err := c.marshaler.Marshal(&p)
		if err != nil {
			return err
		}

		for _, pair := range p.CachePairs() {
			missingCacheKeys = append(missingCacheKeys, c.PropertiesByFieldCacheKey(pair.Key, pair.Value))
			missingValues = append(missingValues, string(marshalled))
		}

		return nil
	}); err != nil {
		return err
	}

	return c.cacheProperties(ctx, overwrite, missingCacheKeys, missingValues)
}

func (c *CacherImpl) cacheProperties(ctx context.Context, overwrite bool, missingCacheKeys, missingValues []string) error {
	bgCtx := backend.DetachContext(ctx)

	var err error
	if overwrite {
		operation := "set"
		err = c.redis.MSetWithTTL(ctx, missingCacheKeys, missingValues, c.cacheTTLMS)
		primaryOperation(ctx, operation, err)
		// Invoke go routine to on backup (To replicate primary behavior)
		go func() {
			bkupErr := c.redisBackup.MSetWithTTL(bgCtx, missingCacheKeys, missingValues, c.cacheTTLMS)
			replicateOperation(bgCtx, operation, bkupErr)
		}()

	} else {
		operation := "safeset"
		_, err = c.redis.MSafeSetWithTTL(ctx, missingCacheKeys, missingValues, c.cacheTTLMS)
		primaryOperation(ctx, operation, err)
		// Invoke go routine to on backup (To replicate primary behavior)
		go func() {
			_, bkupErr := c.redisBackup.MSafeSetWithTTL(bgCtx, missingCacheKeys, missingValues, c.cacheTTLMS)
			replicateOperation(bgCtx, operation, bkupErr)
		}()

	}

	return err
}
