package rediscacher

import (
	"testing"
	"time"

	"code.justin.tv/chat/redis"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type SetTest struct {
	suite.Suite
	mockRedis *redis.MockRedis
	cb        *CacherImpl
	keys      []string
}

func (suite *SetTest) SetupTest() {
	suite.mockRedis = &redis.MockRedis{}
	suite.keys = []string{}
	suite.cb = &CacherImpl{
		redis:       suite.mockRedis,
		redisBackup: suite.mockRedis,
		cacheTTLMS:  time.Minute,
		key:         "user-props-string",
		marshaler:   &JSONMarshaler{},
	}
	for _, pair := range Users[ToBanUserID].CachePairs() {
		suite.keys = append(suite.keys, suite.cb.PropertiesByFieldCacheKey(pair.Key, pair.Value))
	}
}

func (suite *SetTest) TestCacheUsersPropertiesWithOverwrite() {
	mockRedis := suite.mockRedis

	mockRedis.On("MSetWithTTL", mock.Anything, suite.keys, mock.Anything, suite.cb.cacheTTLMS).Return(nil)
	err := suite.cb.CacheProperties(context.Background(), true, *Users[ToBanUserID])
	assert.NoError(suite.T(), err)
	mockRedis.AssertNotCalled(suite.T(), "MSafeSetWithTTL", mock.Anything, mock.Anything, mock.Anything, mock.Anything)
}

func (suite *SetTest) TestCacheUsersPropertiesWithoutOverwrite() {
	mockRedis := suite.mockRedis
	for _, pair := range Users[AdminUserID].CachePairs() {
		suite.keys = append(suite.keys, suite.cb.PropertiesByFieldCacheKey(pair.Key, pair.Value))
	}
	mockRedis.On("MSafeSetWithTTL", mock.Anything, suite.keys, mock.Anything, suite.cb.cacheTTLMS).Return([]bool{true, true}, nil)

	err := suite.cb.BulkSetProperties(context.Background(), false, models.PropertiesIterator([]models.Properties{*Users[ToBanUserID], *Users[AdminUserID]}))
	assert.NoError(suite.T(), err)

	mockRedis.AssertNotCalled(suite.T(), "MSetWithTTL", mock.Anything, mock.Anything, mock.Anything, mock.Anything)
}

func TestSetTest(t *testing.T) {
	suite.Run(t, new(SetTest))
}
