package rediscacher

import (
	"bytes"
	"compress/gzip"
	"encoding/json"
	"errors"
	"io"
	"reflect"
	"sync"

	"code.justin.tv/chat/golibs/errx"
)

type JSONMarshaler struct{}

func (m *JSONMarshaler) Marshal(e interface{}) ([]byte, error) {
	return json.Marshal(e)
}

func (m *JSONMarshaler) Unmarshal(b []byte, e interface{}) error {
	return json.Unmarshal(b, e)
}

type GZipMarshaler struct {
	Marshaler Marshaler
}

func (m *GZipMarshaler) Marshal(e interface{}) ([]byte, error) {
	jb, err := m.Marshaler.Marshal(e)
	if err != nil {
		return nil, err
	}

	var buf bytes.Buffer
	zw := gzip.NewWriter(&buf)

	_, err = zw.Write(jb)
	if err != nil {
		return nil, errx.New(err)
	}

	if err := zw.Close(); err != nil {
		return nil, errx.New(err)
	}

	return buf.Bytes(), nil
}

func (m *GZipMarshaler) Unmarshal(b []byte, e interface{}) error {
	var zbuf bytes.Buffer
	if _, err := zbuf.Write(b); err != nil {
		return errx.New(err)
	}

	var buf bytes.Buffer
	zr, err := gzip.NewReader(&zbuf)
	if err != nil {
		return errx.New(err)
	}
	_, err = io.Copy(&buf, zr)
	if err != nil {
		return errx.New(err)
	}

	return m.Marshaler.Unmarshal(buf.Bytes(), e)
}

func unmarshalResults(marshaler Marshaler, cacheResults []string, e interface{}) ([]int, []int, error) {
	t := reflect.TypeOf(e)
	if t.Kind() != reflect.Ptr {
		return nil, nil, errors.New("must be pointer")
	}
	t = t.Elem()

	if t.Kind() != reflect.Slice {
		return nil, nil, errors.New("must be slice")
	}

	v := reflect.ValueOf(e).Elem()

	var misses []int
	var deletes []int

	// Keep a different index because some cacheResults will be empty
	// and excluded from the resulting array
	hitIndex := 0
	for i, cacheResult := range cacheResults {
		if cacheResult == "" {
			misses = append(misses, i)
			continue
		}

		// Taken from json.Unmarshal
		if hitIndex >= v.Cap() {
			newcap := v.Cap() + v.Cap()/2
			if newcap < 4 {
				newcap = 4
			}
			newv := reflect.MakeSlice(v.Type(), v.Len(), newcap)
			reflect.Copy(newv, v)
			v.Set(newv)
		}

		// Taken from json.Unmarshal
		if hitIndex >= v.Len() {
			v.SetLen(hitIndex + 1)
		}

		item := v.Index(hitIndex).Interface()
		// Get new struct otherwise json.Unmarshal changes the type to a map
		// if the struct is nil
		newItem := reflect.New(reflect.TypeOf(item)).Interface()
		if err := marshaler.Unmarshal([]byte(cacheResult), newItem); err != nil {
			misses = append(misses, i)
			deletes = append(deletes, i)
			continue
		}
		v.Index(hitIndex).Set(reflect.ValueOf(newItem).Elem())
		hitIndex++
	}

	// Taken from json.Unmarshal
	if hitIndex < v.Len() {
		v.SetLen(hitIndex)
	}

	if hitIndex < v.Cap() {
		v.SetCap(hitIndex)
	}

	reflect.ValueOf(e).Elem().Set(v)

	return misses, deletes, nil
}

type cachee struct {
	index  int
	object string
}

type marshaledCachee struct {
	index  int
	object interface{}
}

func ccUnmarshalResults(marshaler Marshaler, cacheResults []string, e interface{}) ([]int, []int, error) {
	if len(cacheResults) == 0 {
		return nil, nil, nil
	}

	t := reflect.TypeOf(e)
	if t.Kind() != reflect.Ptr {
		return nil, nil, errors.New("must be pointer")
	}
	t = t.Elem()

	if t.Kind() != reflect.Slice {
		return nil, nil, errors.New("must be slice")
	}

	v := reflect.ValueOf(e).Elem()

	workers := 10

	var misses []int
	var deletes []int

	missesCh := make(chan int, workers)
	deletesCh := make(chan int, workers)
	cachesCh := make(chan cachee, workers)
	hitsCh := make(chan marshaledCachee, workers)

	if len(cacheResults) >= v.Cap() {
		newv := reflect.MakeSlice(v.Type(), len(cacheResults), len(cacheResults))
		reflect.Copy(newv, v)
		v.Set(newv)
	}

	itemInterface := v.Index(0).Interface()

	var cacheWg sync.WaitGroup
	cacheWg.Add(workers)

	for i := 0; i < workers; i++ {
		go func() {
			defer cacheWg.Done()
			for result := range cachesCh {
				// Get new struct otherwise json.Unmarshal changes the type to a map
				// if the struct is nil
				newItem := reflect.New(reflect.TypeOf(itemInterface)).Interface()
				if err := marshaler.Unmarshal([]byte(result.object), newItem); err != nil {
					missesCh <- i
					deletesCh <- i
					continue
				}

				hitsCh <- marshaledCachee{
					index:  result.index,
					object: newItem,
				}
			}
		}()
	}

	var processWg sync.WaitGroup
	processWg.Add(1)
	go func() {
		defer processWg.Done()
		for {
			if missesCh == nil && deletesCh == nil && hitsCh == nil {
				break
			}

			select {
			case miss, ok := <-missesCh:
				if !ok {
					missesCh = nil
					continue
				}
				misses = append(misses, miss)
			case delete, ok := <-deletesCh:
				if !ok {
					deletesCh = nil
					continue
				}
				deletes = append(deletes, delete)
			case hit, ok := <-hitsCh:
				if !ok {
					hitsCh = nil
					continue
				}
				v.Index(hit.index).Set(reflect.ValueOf(hit.object).Elem())
			}
		}
	}()

	// Keep a different index because some cacheResults will be empty
	// and excluded from the resulting array
	hitIndex := 0

	for i, cacheResult := range cacheResults {
		if cacheResult == "" {
			missesCh <- i
			continue
		}

		cachesCh <- cachee{
			index:  hitIndex,
			object: cacheResult,
		}
		hitIndex++
	}

	close(cachesCh)
	cacheWg.Wait()
	close(hitsCh)
	close(missesCh)
	close(deletesCh)
	processWg.Wait()

	// Taken from json.Unmarshal
	if hitIndex < v.Len() {
		v.SetLen(hitIndex)
	}

	if hitIndex < v.Cap() {
		v.SetCap(hitIndex)
	}

	reflect.ValueOf(e).Elem().Set(v)

	return misses, deletes, nil
}
