package rediscacher

import (
	"encoding/json"
	"testing"

	"fmt"

	"code.justin.tv/web/users-service/models"
)

type unmarshalFunc func(Marshaler, []string, interface{}) ([]int, []int, error)

func TestBulkUnmarshal(t *testing.T) {
	for name, scenario := range map[string]struct {
		Marshaler Marshaler

		UnmarshalFunc unmarshalFunc
	}{
		"bulk-json": {
			Marshaler:     &JSONMarshaler{},
			UnmarshalFunc: bulkUnmarshalResults,
		},
		"reflect-json": {
			Marshaler:     &JSONMarshaler{},
			UnmarshalFunc: unmarshalResults,
		},
		"reflect-gzip-json": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: unmarshalResults,
		},
	} {
		t.Run(name, func(t *testing.T) {
			users := []models.Properties{
				{
					ID: "1",
				},
				{
					ID: "2",
				},
				{
					ID: "3",
				},
			}

			cacheResults := make([]string, len(users))
			for i, user := range users {
				b, err := scenario.Marshaler.Marshal(user)
				if err != nil {
					t.Fatal("failed to marshal user:", err)
				}

				cacheResults[i] = string(b)
			}

			var props []models.Properties
			misses, deletes, err := scenario.UnmarshalFunc(scenario.Marshaler, cacheResults, &props)
			if err != nil {
				t.Error(err)
			}

			for i, prop := range props {
				if fmt.Sprint(i+1) != prop.ID {
					t.Errorf("id mismatch: %d != %s", i+1, prop.ID)
				}
			}

			if len(misses) != 0 {
				t.Error("misses", len(misses))
			}

			if len(deletes) != 0 {
				t.Error("deletes", len(deletes))
			}
		})
	}
}

func BenchmarkUnmarshalFunc(b *testing.B) {
	for name, scenario := range map[string]struct {
		Marshaler     Marshaler
		UnmarshalFunc unmarshalFunc
		Size          int
	}{
		"bulk-json-100": {
			Marshaler:     &JSONMarshaler{},
			UnmarshalFunc: bulkUnmarshalResults,
			Size:          100,
		},
		"reflect-json-100": {
			Marshaler:     &JSONMarshaler{},
			UnmarshalFunc: unmarshalResults,
			Size:          100,
		},
		"reflect-gzip-json-100": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: unmarshalResults,
			Size:          100,
		},
		"bulk-json-1000": {
			Marshaler:     &JSONMarshaler{},
			UnmarshalFunc: bulkUnmarshalResults,
			Size:          1000,
		},
		"reflect-json-1000": {
			Marshaler:     &JSONMarshaler{},
			UnmarshalFunc: unmarshalResults,
			Size:          1000,
		},
		"reflect-gzip-json-1000": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: unmarshalResults,
			Size:          1000,
		},
		"cc-reflect-json-100": {
			Marshaler:     &JSONMarshaler{},
			UnmarshalFunc: unmarshalResults,
			Size:          100,
		},
		"cc-reflect-gzip-json-100": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: ccUnmarshalResults,
			Size:          100,
		},
		"cc-reflect-json-1000": {
			Marshaler:     &JSONMarshaler{},
			UnmarshalFunc: ccUnmarshalResults,
			Size:          1000,
		},
		"cc-reflect-gzip-json-1000": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: ccUnmarshalResults,
			Size:          1000,
		},
		"reflect-gzip-json-1": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: ccUnmarshalResults,
			Size:          1,
		},
		"reflect-gzip-json-10": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: unmarshalResults,
			Size:          10,
		},
		"cc-reflect-gzip-json-1": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: unmarshalResults,
			Size:          1,
		},
		"cc-reflect-gzip-json-10": {
			Marshaler: &GZipMarshaler{
				Marshaler: &JSONMarshaler{},
			},
			UnmarshalFunc: ccUnmarshalResults,
			Size:          10,
		},
	} {
		b.Run(name, func(b *testing.B) {
			benchmarkFuncMarshaler(b, scenario.UnmarshalFunc, scenario.Marshaler, scenario.Size)
		})
	}
}

func benchmarkFuncMarshaler(b *testing.B, f unmarshalFunc, m Marshaler, size int) {
	expected := make([]string, size)

	for i := 0; i < size; i++ {
		mb, err := m.Marshal(&models.Properties{
			ID: fmt.Sprintf("%d", i),
		})
		if err != nil {
			b.Fatal(err)
		}
		expected[i] = string(mb)
	}

	for i := 0; i < b.N; i++ {
		var props []models.Properties

		_, _, err := f(m, expected, &props)
		if err != nil {
			b.Fatal(err)
		}
	}
}

func BenchmarkMarshal(b *testing.B) {
	for name, marshaler := range map[string]Marshaler{
		"gzip-json": &GZipMarshaler{
			Marshaler: &JSONMarshaler{},
		},
		"json": &JSONMarshaler{},
	} {
		prop := models.Properties{
			ID: "1",
		}
		b.Run(name, func(b *testing.B) {
			for i := 0; i < b.N; i++ {
				if _, err := marshaler.Marshal(prop); err != nil {
					b.Fatal(err)
				}
			}
		})
	}
}

func BenchmarkUnmarshal(b *testing.B) {
	for name, marshaler := range map[string]Marshaler{
		"gzip-json": &GZipMarshaler{
			Marshaler: &JSONMarshaler{},
		},
		"json": &JSONMarshaler{},
	} {
		mb, err := marshaler.Marshal(models.Properties{
			ID: "1",
		})
		if err != nil {
			b.Fatal(err)
		}
		b.Run(name, func(b *testing.B) {
			for i := 0; i < b.N; i++ {
				var prop models.Properties
				if err := marshaler.Unmarshal(mb, &prop); err != nil {
					b.Fatal(err)
				}
			}
		})
	}
}

func BenchmarkMarshalSize(b *testing.B) {
	for name, marshaler := range map[string]Marshaler{
		"gzip-json": &GZipMarshaler{
			Marshaler: &JSONMarshaler{},
		},
		"json": &JSONMarshaler{},
	} {
		var prop models.Properties
		if err := json.Unmarshal([]byte(user), &prop); err != nil {
			b.Fatal(err)
		}

		b.Run(name, func(b *testing.B) {
			for i := 0; i < b.N; i++ {
				mb, err := marshaler.Marshal(prop)
				if err != nil {
					b.Fatal(err)
				}

				b.Log(len(mb))
			}
		})
	}
}

const user = `{
    "id":"51756334",
    "login":"mellified_man",
    "birthday":"1927-03-03T00:00:00Z",
    "dmca_violation":null,
    "terms_of_service_violation":null,
    "deleted_on":null,
    "language":"en",
    "category":"gaming",
    "remote_ip":"231.231.231.231",
    "email":"mellified_man@fakeemail.com",
    "last_login":"2016-11-30 19:35:07",
    "banned_until":null,
    "dmca_violation_count":null,
    "tos_violation_count":null,
    "admin":true,
    "subadmin":false,
    "global_mod":false,
    "displayname":"Mellified_Man",
    "description":"Director of Infrastructure Engineering @Curse (a division of Twitch!), homeschool dad of 3 great boys, avid gamer, happy tech nerd, grumpy libertarian, Blizzard \u0026 Stardock veteran",
    "profile_image":{
      "150x150":{
        "width":150,
        "height":150,
        "url":"https://static-cdn.jtvnw.net/jtv_user_pictures/mellified_man-profile_image-e801edf78b30c43e-150x150.jpeg",
        "uid":"e801edf78b30c43e",
        "size":"150x150",
        "format":"jpeg"
      },
      "28x28":{
        "width":28,
        "height":28,
        "url":"https://static-cdn.jtvnw.net/jtv_user_pictures/mellified_man-profile_image-e801edf78b30c43e-28x28.jpeg",
        "uid":"e801edf78b30c43e",
        "size":"28x28",
        "format":"jpeg"
      },
      "300x300":{
        "width":300,
        "height":300,
        "url":"https://static-cdn.jtvnw.net/jtv_user_pictures/mellified_man-profile_image-e801edf78b30c43e-300x300.jpeg",
        "uid":"e801edf78b30c43e",
        "size":"300x300",
        "format":"jpeg"
      },
      "50x50":{
        "width":50,
        "height":50,
        "url":"https://static-cdn.jtvnw.net/jtv_user_pictures/mellified_man-profile_image-e801edf78b30c43e-50x50.jpeg",
        "uid":"e801edf78b30c43e",
        "size":"50x50",
        "format":"jpeg"
      },
      "600x600":{
        "width":600,
        "height":600,
        "url":"https://static-cdn.jtvnw.net/jtv_user_pictures/mellified_man-profile_image-e801edf78b30c43e-600x600.jpeg",
        "uid":"e801edf78b30c43e",
        "size":"600x600",
        "format":"jpeg"
      },
      "70x70":{
        "width":70,
        "height":70,
        "url":"https://static-cdn.jtvnw.net/jtv_user_pictures/mellified_man-profile_image-e801edf78b30c43e-70x70.jpeg",
        "uid":"e801edf78b30c43e",
        "size":"70x70",
        "format":"jpeg"
      }
    },
    "updated_on":"2016-12-09T18:30:27.082867Z",
    "created_on":"2013-11-18T03:05:31.52863Z",
    "email_verified":true,
    "phone_number":"5557969462",
    "last_login_change_date":null
}`
