package warmercacher

import (
	"fmt"

	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/models"
)

// WarmerCacher is a Cacher that handles cache operations
// using a main cache and warms another cache with set operations.
type WarmerCacher struct {
	main backend.Cacher
	warm backend.Cacher

	// decides if reads should be written
	writeDecider Decider
	// decides if reads should go to warmup
	readDecider Decider
}

func New(main, warm backend.Cacher, write, read Decider) *WarmerCacher {
	return &WarmerCacher{
		main: main,
		warm: warm,

		writeDecider: write,
		readDecider:  read,
	}
}

// GetProperties reads from warmup if it is decided. Otherwise, gets properties
// from main and writes them to warmup if it is decided.
func (c *WarmerCacher) GetProperties(ctx context.Context, field string, key string, e interface{}) error {
	if c.readDecider.Decide() {
		err := c.warm.GetProperties(ctx, field, key, e)
		if err == nil || backend.IsCacheMissErr(err) {
			return err
		}
		logx.Error(ctx, fmt.Sprint("failed to read from warmup cache:", err))
	}
	// Get from main cache
	err := c.main.GetProperties(ctx, field, key, e)

	if c.writeDecider.Decide() && err == nil {
		bgCtx := backend.DetachContext(ctx)
		go func() {
			// Potentially send it to warm other cache
			cacheable, ok := e.(models.Cacheable)
			if !ok {
				return
			}

			if err := c.warm.CacheProperties(bgCtx, true, cacheable); err != nil {
				logx.Error(ctx, fmt.Sprint("failed to set props in warmup cache:", err))
			}
		}()
	}

	return err
}

// ExpireProperties expires the property in both caches. A failure in the warmup
// will not be returned.
func (c *WarmerCacher) ExpireProperties(ctx context.Context, prop models.Cacheable) error {
	var g errgroup.Group

	g.Go(func() error {
		return c.main.ExpireProperties(ctx, prop)
	})

	// Always expire warmup
	g.Go(func() error {
		if err := c.warm.ExpireProperties(ctx, prop); err != nil {
			logx.Error(ctx, fmt.Sprint("failed to expire props in warmup cache:", err))
		}

		return nil
	})

	return g.Wait()
}

// CacheProperties caches properties in the main and warmup cache.
func (c *WarmerCacher) CacheProperties(ctx context.Context, overwrite bool, props models.Cacheable) error {
	var g errgroup.Group

	g.Go(func() error {
		return c.main.CacheProperties(ctx, overwrite, props)
	})

	g.Go(func() error {
		if err := c.warm.CacheProperties(ctx, overwrite, props); err != nil {
			logx.Error(ctx, fmt.Sprint("failed to set props in warmup cache:", err))
		}
		return nil
	})

	return g.Wait()
}

// BulkGetProperties reads from warmup if it is decided. Otherwise, bulk gets
// from main and then writes them to warmup if it is decided.
func (c *WarmerCacher) BulkGetProperties(ctx context.Context, field string, key []string, e interface{}) ([]int, error) {
	if c.readDecider.Decide() {
		missing, err := c.warm.BulkGetProperties(ctx, field, key, e)
		if err == nil || backend.IsCacheMissErr(err) {
			return missing, err
		}
		logx.Error(ctx, fmt.Sprint("failed to bulk read from warmup cache:", err))
	}

	missing, err := c.main.BulkGetProperties(ctx, field, key, e)

	if c.writeDecider.Decide() && err == nil {
		bgCtx := backend.DetachContext(ctx)
		go func() {
			cacheIter, ok := e.(models.CacheableIterator)
			if !ok {
				return
			}
			if err := c.warm.BulkSetProperties(bgCtx, true, cacheIter); err != nil {
				logx.Error(ctx, fmt.Sprint("failed to bulk set props in warmup cache:", err))
			}
		}()
	}

	return missing, err
}

// BulkSetProperties sets properties in the main cache and warmup cache.
func (c *WarmerCacher) BulkSetProperties(ctx context.Context, overwrite bool, iter models.CacheableIterator) error {
	var g errgroup.Group

	g.Go(func() error {
		return c.main.BulkSetProperties(ctx, overwrite, iter)
	})

	g.Go(func() error {
		if err := c.warm.BulkSetProperties(ctx, overwrite, iter); err != nil {
			logx.Error(ctx, fmt.Sprint("failed to bulk set props in warmup cache:", err))
		}
		return nil
	})

	return g.Wait()
}
