package discovery

import (
	"net/http"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/web/discovery/client"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/models"
	"github.com/afex/hystrix-go/hystrix"
	"golang.org/x/net/context"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"discovery_get_game_by_name": {
			Timeout:               1000,
			MaxConcurrentRequests: 10000,
		},
	})
}

const (
	DiscoveryErr = "Discovery Error"
)

var ErrGameNotFound = &models.CodedError{
	ErrorValue:      "Game not found.",
	CodeValue:       "error_game_not_found",
	StatusCodeValue: http.StatusBadRequest,
}

type Discovery interface {
	GetGameIDByName(ctx context.Context, game string) (int, error)
}

type discoveryImpl struct {
	client discovery.Client
}

func NewDiscovery(clientConf twitchhttp.ClientConf) (Discovery, error) {
	c, err := discovery.NewClient(clientConf)
	if err != nil {
		return nil, err
	}
	return &discoveryImpl{c}, err
}

func (d *discoveryImpl) GetGameIDByName(ctx context.Context, game string) (int, error) {
	var gameID int
	hystrixErr := hystrix.Do("discovery_get_game_by_name", func() error {
		var id int
		id, e := d.get(ctx, game)
		if e == ErrGameNotFound {
			gameID = id
			return nil
		} else {
			gameID = id
			return e
		}
	}, nil)

	if hystrixErr != nil && !utils.IsHystrixErr(hystrixErr) {
		logx.Error(ctx, "Discovery Internal Error", logx.Fields{DiscoveryErr: hystrixErr.Error()})
	}

	return gameID, hystrixErr
}

func (d *discoveryImpl) get(ctx context.Context, game string) (int, error) {
	g, err := d.client.GetByName(ctx, game, "", nil)
	if g != nil && g.ID != 0 {
		return g.ID, nil
	} else if err != nil && (g == nil || g.ID == 0) {
		return 0, ErrGameNotFound
	} else {
		return 0, err
	}
}
