package hystrix

import (
	"fmt"
	"log"

	"code.justin.tv/common/config"
	"code.justin.tv/web/users-service/configs"

	"github.com/afex/hystrix-go/hystrix"
	metricCollector "github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/afex/hystrix-go/plugins"
)

const (
	servicePrefix = "users-service"
	// CommandHideAllFollows is a hystrix command to hide all of a user's follows
	CommandHideAllFollows = "follows_hide_all_follows"
	// CommandRestoreAllFollows is a hystrix command to unhide all of a user's follows
	CommandRestoreAllFollows = "follows_restore_all_follows"
	// DefaultTimeout is a default timeout, in milliseconds, for a hystrix command
	DefaultTimeout = 3000
	// DefaultSleepWindow is the default amount of time, in milliseconds, to sleep until recovery is checked
	DefaultSleepWindow = 5000
	// DefaultErrorThreshold is a default threshold of errors required for a circuit to open
	DefaultErrorThreshold = 50
	// DefaultRequestThreshold is the default minimum amount of requests needed before a circuit can trip
	DefaultRequestThreshold = 20
	// DefaultMaxConcurrency is a default for many instances of a command can run concurrently
	DefaultMaxConcurrency = 100
)

// InitHystrix configures hystrix commands.
func InitHystrix(sampling, errorSampling float32) {
	defaultConfig := hystrix.CommandConfig{
		Timeout:                DefaultTimeout,
		SleepWindow:            DefaultSleepWindow,
		ErrorPercentThreshold:  DefaultErrorThreshold,
		RequestVolumeThreshold: DefaultRequestThreshold,
		MaxConcurrentRequests:  DefaultMaxConcurrency,
	}

	hystrix.ConfigureCommand(CommandHideAllFollows, defaultConfig)

	hystrix.ConfigureCommand(CommandRestoreAllFollows, defaultConfig)

	statsdHostPort := config.StatsdHostPort()
	statsdPrefix := fmt.Sprintf("%s.%s.hystrix", servicePrefix, configs.EnvironmentRegion())
	log.Printf("Hystrix is connecting to statsd at %s with a prefix of %s", statsdHostPort, statsdPrefix)
	c, err := plugins.InitializeStatsdCollector(&plugins.StatsdCollectorConfig{
		StatsdAddr:      statsdHostPort,
		Prefix:          statsdPrefix,
		SampleRate:      sampling,
		ErrorSampleRate: errorSampling,
	})
	if err != nil {
		log.Fatal(err)
		return
	}
	metricCollector.Registry.Register(c.NewStatsdCollector)
}

// Do implements a straight passthrough to hystrix singleton
func Do(name string, run func() error, fallback func(error) error) error {
	return hystrix.Do(name, run, fallback)
}
