package kinesis

import (
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/identity/publisher"
)

// Publisher publishes an event to a Kinesis stream, logging any errors that may
// have occurred
type Publisher interface {
	Publish(ctx context.Context, e Event)
}

type publisherImpl struct {
	client publisher.Publisher
}

// NewPublisher instantiates a new Publisher
func NewPublisher(awsRegion string, retryCount int, retryDelay int, roleARN string, streamName string, stats statsd.Statter) (Publisher, error) {
	config := &publisher.Config{
		AWSRegion:       awsRegion,
		RetryCount:      retryCount,
		RetryDelay:      time.Duration(retryDelay) * time.Millisecond,
		RoleARN:         roleARN,
		StatsSampleRate: 1.0,
		StreamName:      streamName,
	}
	pub, err := publisher.NewPublisher(config, stats)
	if err != nil {
		return nil, err
	}
	return &publisherImpl{client: pub}, nil
}

// Publish attempts to publish the given event to the Kinesis stream.
// Errors during publish will be reported to Rollbar.
func (p *publisherImpl) Publish(ctx context.Context, e Event) {
	err := p.client.Publish(ctx, e)
	if err != nil {
		logx.Error(ctx, err)
	}
}
